UPM.define('LicenseDetailsView',
    [
        'jquery',
        'BaseView',
        'LicenseDetailsTemplate',
        'LicenseTokenTemplate',
        'ManageAddonFlows',
        'UpmEnvironment',
        'UpmFormats',
        'UpmLicenseInfo',
        'UpmStrings'
    ], function($,
                BaseView,
                licenseDetailsTemplate,
                licenseTokenTemplate,
                ManageAddonFlows,
                UpmEnvironment,
                UpmFormats,
                UpmLicenseInfo,
                UpmStrings) {

    "use strict";

    /**
     * Renders the license detail area within the add-on details, and allows editing of the license
     * and/or license token if applicable.  The model for this view is the add-on model.
     */
    return BaseView.extend({
        events: {
            'click input.submit': '_onSubmitLicense',
            'click a.upm-plugin-license-edit': '_toggleLicenseEdit',
            'click a.upm-license-cancel': '_toggleLicenseEdit',
            'click .upm-plugin-license-truncated': '_toggleFullLicenseKey',
            'click .edit-token': '_onStartLicenseTokenEdit',
            'click .cancel-edit-token': '_onStopLicenseTokenEdit',
            'click .generate-token': '_onGenerateToken',
            'click .save-token': '_onSaveToken',
            'change .token-state': '_onChangeTokenState',
            'keyup .token-editable': '_onTokenFieldChange',
            'input .token-editable': '_onTokenFieldChange',
            'propertychange .token-editable': '_onTokenFieldChange'
        },

        _getHtml: function() {
            var licenseDetailsHtml,
                licenseTokenHtml,
                plugin = this.model.toJSON(),
                license = plugin.licenseDetails;

            if (this.model.isLicenseUpdatable() || this.model.getLicenseDetails()) {
                licenseDetailsHtml = licenseDetailsTemplate({
                    plugin: plugin,
                    license: license,
                    readOnly: plugin.licenseReadOnly,
                    description: UpmLicenseInfo.getLicenseDescription(license),
                    status: license && UpmLicenseInfo.getLicenseStatusDescription(plugin, license),
                    unlicensed: !license || UpmLicenseInfo.isUnlicensed(license)
                });
            }

            if (this.model.canHaveAccessToken()) {
                licenseTokenHtml = licenseTokenTemplate({
                    token: this.model.getAccessToken(),
                    tokenValue: this.model.getAccessTokenValue(),
                    allowSelection: this.model.isPaidViaAtlassian()
                });
            }

            return $('<div></div>').append(licenseDetailsHtml).append(licenseTokenHtml);
        },

        _initEvents: function() {
            this.listenTo(this.model, 'change:accessToken', this.render);
            this.listenTo(this.model, 'change:licenseDetails', this.render);
            this.listenTo(this.model, 'uninstalled', this._onUninstalled);
        },

        _getLicenseTokenContainer: function() {
            return this.$el.find('.upm-plugin-license-token-container');
        },

        _getSelectedTokenState: function() {
            return this.$el.find('select.token-state').val();
        },

        _getTokenValue: function() {
            return this.model.getAccessTokenValue();
        },

        _postRender: function() {
            var initialLicenseValue = (this.model.getLicenseDetails() && this.model.getLicenseDetails().rawLicense) || '',
                $formButton = this.$el.find('input.submit'),
                $textarea = this.$el.find('textarea');

            // Disable the update button unless the value of the textarea changes
            $textarea.bind('keyup input change propertychange', function() {
                // Level the playing field for line breaks. Replace all double line breaks (windows) with single (unix)
                $formButton.prop('disabled',
                    ($textarea.val().replace(/\r\n/g, '\n') === initialLicenseValue.replace(/\r\n/g, '\n')));
            });

            if (this.model.getAccessToken()) {
                var initialTokenValue = this.model.getAccessTokenValue(),
                    $tokenEdit = this.$el.find('.token-editable'),
                    $tokenSave = this.$el.find('.save-token');

                this.$el.find('select.token-state').val(this.model.getAccessToken().state);

                // Disable the save button unless the value of the token changes
                $tokenEdit.bind('keyup input change propertychange', function() {
                    $tokenSave.prop('disabled', ($.trim($tokenEdit.val() || '') === initialTokenValue));
                });
            }
        },

        _onSubmitLicense: function(e) {
            var $licenseField = this.$el.find('.edit-license-key'),
                $button = this.$el.find('.submit-license'),
                $spinner = this.$el.find('.upm-license-form .loading'),
                licenseKey = $licenseField.val().trim(),
                me = this;

            e.preventDefault();

            $licenseField.prop('disabled', true);
            $button.prop('disabled', true);
            $spinner.removeClass('hidden');

            this.model.updateLicense(licenseKey)
                .fail(function(xhr) {
                    // clear the busy state only if there's an error, since if we succeeded,
                    // we would have refreshed the whole view state
                    $licenseField.prop('disabled', false);
                    $button.prop('disabled', false);
                    $spinner.addClass('hidden');

                    me.model.signalAjaxError(xhr);
                })
                .always(function() {
                    UPM.trace('license-updated');
                });
        },

        _onTokenFieldChange: function(e) {
            var valueUnchanged = ((this.$el.find('.token-editable').val() || '') === this._getTokenValue());
            this.$el.find('.save-token').toggleClass('disabled', valueUnchanged).attr('aria-disabled', valueUnchanged);
        },

        _onUninstalled: function() {
            this.$el.addClass('disabled');
            this.$el.find('input[type="submit"]').attr('disabled', 'disabled');
            this.$el.find('textarea').attr('disabled', 'disabled');
        },

        _toggleFullLicenseKey: function(e) {
            e.preventDefault();
            this.$el.find('.upm-plugin-license-truncated').toggleClass('hidden');
            this.$el.find('.upm-plugin-license-raw').toggleClass('hidden');
        },

        _toggleLicenseEdit: function(e) {
            var $container = this.$el.find('div.upm-license-details');
            e.preventDefault();
            if ($container.hasClass('edit-license')) {
                $container.removeClass('edit-license');
            } else {
                $container.addClass('edit-license');
                $container.find('form.upm-license-form textarea').focus().select();
            }            
        },

        _onStartLicenseTokenEdit: function(e) {
            e.preventDefault();
            this._toggleEditableView(true);
        },

        _onStopLicenseTokenEdit: function(e) {
            e.preventDefault();
            this._toggleEditableView(false);
            this.$el.find('.token-editable').val(this._getTokenValue());
            this.$el.find('.save-token').prop('disabled', true);
        },

        _onGenerateToken: function(e) {
            ManageAddonFlows.generateLicenseToken(this.model);
        },

        _onSaveToken: function(e) {
            var me = this,
                tokenVal = this.$el.find('.token-editable').val().trim();

            e.preventDefault();
            if (this.$el.find('.save-token').hasClass('disabled')) {
                return;
            }
            
            this._setBusy(true);

            var isUrl = function(s) {
                if (/^(http|https|ftp):\/\/[a-z0-9]+([\-\.]{1}[a-z0-9]+)*\.[a-z]{2,5}(:[0-9]{1,5})?(\/.*)?$/i.test(s)) {
                    return true;
                }
                return false;
            };
            
            if (tokenVal && isUrl(tokenVal)) {
                tokenVal = UpmEnvironment.getUrlParam('access-token', tokenVal);
            }

            this.model.updateAccessToken(tokenVal, tokenVal && this._getSelectedTokenState())
                .always(function() {
                    me._setBusy(false);
                });
        },

        _onChangeTokenState: function(e) {
            var me = this;

            this._setBusy(true, true);

            this.model.updateAccessToken(this._getTokenValue(), this._getSelectedTokenState())
                .always(function() {
                    me._setBusy(false, true);
                });
        },

        _setBusy: function(busy, spinnerOnStateField) {
            var $spinner = this.$el.find(spinnerOnStateField ?
                '.upm-plugin-license-token-state-container .loading' :
                '.upm-plugin-license-token .loading');
            this.$el.find('.generate-token').prop('disabled', busy);
            this.$el.find('.token-state').prop('disabled', busy);
            $spinner.toggleClass('hidden', !busy);
        },

        _toggleEditableView: function(editable) {
            var tokenVal = this._getTokenValue();
            this.$el.find('.token-editable, .save-token, .cancel-edit-token').toggleClass('hidden', !editable);
            this.$el.find('.edit-token').toggleClass('hidden', editable);
            this.$el.find('.token-readonly').toggleClass('hidden', editable || !tokenVal);
            this.$el.find('.generate-token').toggleClass('hidden', editable || tokenVal);
        }
    });
});
