define("workflow-designer/looped-transition-figure", [
    "workflow-designer/triangle-figure-upwards",
    "workflow-designer/layout/locator/static-locator",
    "workflow-designer/draw-2d",
    "workflow-designer/underscore",
    "jquery"
], function(
    TriangleFigureUpwards,
    StaticLocator,
    draw2d,
    _,
    jQuery
) {
    return draw2d.shape.basic.Rectangle.extend(
        /** @lends JIRA.WorkflowDesigner.LoopedTransitionFigure# */
        {
            /**
             * @constructs
             * @classdesc `LoopedTransitionView`'s Draw2D figure.
             * @extends draw2d.shape.basic.Rectangle
             */
            init: function (transitionModel) {
                var arrow, hover, lozenge, lozengeLocator, loopRectangle, loopRectangleLocator, nameLozenge, nameLozengeLocator;

                _.bindAll(this, "_onDoubleClick");
                this._super.apply(this, arguments);
                this._transitionModel = transitionModel;

                this.installEditPolicy(new draw2d.policy.figure.SelectionFeedbackPolicy());
                this.setAlpha(0);
                this.setDeleteable(false);
                this.setDimension(90, 40);
                this.setResizeable(false);

                // Create the loop as a rectangle.
                loopRectangle = this._loopRectangle = new draw2d.shape.basic.Rectangle(80, 35);
                loopRectangle.setBackgroundColor("#ffffff");
                loopRectangle.setStroke(1);
                loopRectangle.onDoubleClick = this._onDoubleClick;
                loopRectangle.setColor("#cccccc");
                loopRectangle.setStroke(1);
                loopRectangleLocator = new StaticLocator(this, 0, 0);
                this.addFigure(loopRectangle, loopRectangleLocator);

                arrow = this._arrow = new TriangleFigureUpwards();
                arrow.onDoubleClick = this._onDoubleClick;
                arrow.setDimension(8,9);
                arrow.setBackgroundColor("#cccccc");
                arrow.setColor("#cccccc");
                arrow.setStroke(1);
                this.addFigure(arrow, new StaticLocator(this, -5, 11));

                lozenge = this._createLozenge();
                lozengeLocator = new StaticLocator(this, -10, -5);
                this.addFigure(lozenge, lozengeLocator);

                nameLozenge = this._createNameLozenge();
                nameLozengeLocator = new StaticLocator(this, 20, 27);
                this.addFigure(nameLozenge, nameLozengeLocator);

                hover = this._hover = new draw2d.shape.basic.Rectangle(19, 14);
                hover.onDoubleClick = this._onDoubleClick;
                hover.select = this._onSelect;
                hover.setAlpha(0);
                this.addFigure(hover, nameLozengeLocator);
            },

            /**
             * @return {draw2d.Figure} The "All" lozenge figure.
             * @private
             */
            _createLozenge: function () {
                var rectangle, text;

                rectangle = this._lozengeRectangle = new draw2d.shape.basic.Rectangle(19, 14);
                rectangle.setBackgroundColor("#333333");
                rectangle.setStroke(0);

                text = this._lozengeText = new draw2d.shape.basic.Label();
                text.setAlpha(0);
                text.setBold(true);
                text.setFontColor("#ffffff");
                text.setFontFamily("Arial");
                text.setFontSize(11);
                text.setStroke(0);
                text.setText(AJS.I18n.getText("common.words.all"));
                rectangle.addFigure(text, new draw2d.layout.locator.CenterLocator(rectangle));

                return rectangle;
            },

            _createEllipsifiedName: function (transitionName) {
                if(transitionName.length > 12) {
                    transitionName = transitionName.substring(0, 11)+"..";
                }
                return transitionName;
            },

            /**
             * @return {draw2d.Figure} The "All" lozenge figure.
             * @private
             */
            _createNameLozenge: function () {
                var rectangle, text;

                rectangle = this._nameLozengeRectangle = new draw2d.shape.basic.Rectangle(30, 14);
                rectangle.setBackgroundColor("#ffffff");
                rectangle.setStroke(0);

                text = this._nameLozengeText = new draw2d.shape.basic.Label();
                text.setAlpha(0);
                text.setBold(true);
                text.setFontColor("#333333");
                text.setFontFamily("Arial");
                text.setFontSize(11);
                text.setStroke(0);
                rectangle.addFigure(text, new draw2d.layout.locator.CenterLocator(rectangle));

                return rectangle;
            },

            /**
             * @return {element} The element that triggers a dialog on hover.
             */
            getHoverElement: function () {
                return this._hover.shape.node;
            },

            /**
             * @return {draw2d.util.ArrayList} The figure's ports.
             */
            getPorts: function () {
                // Draw2D's line figure doesn't implement `getPorts()` which causes
                // JS exceptions here. We don't have ports, so return an empty list.
                return new draw2d.util.ArrayList();
            },

            loadText: function() {
                this._nameLozengeText.setText(this._createEllipsifiedName(this._transitionModel.get("name")));
                this._resizeNameLozenge();
            },

            // Draw2D has a bug where only draggable figures are selectable. We
            // don't want global transitions to be draggable, thus this tomfoolery.
            onDrag: jQuery.noop,

            /**
             * Call the figure's `onDoubleClick` handler.
             *
             * @private
             */
            _onDoubleClick: function () {
                this.onDoubleClick.apply(this, arguments);
                this.select();
            },

            /**
             * Resize the "All" lozenge to ensure it contains its text.
             *
             * @private
             */
            _resizeLozenge: function () {
                var textWidth = this._lozengeText.getWidth();

                this._hover.setDimension(textWidth, this._hover.getHeight());
                this._lozengeRectangle.setDimension(textWidth, this._lozengeRectangle.getHeight());
            },

            /**
             * Resize the "name" lozenge to ensure it contains its text.
             *
             * @private
             */
            _resizeNameLozenge: function () {
                var textWidth = this._nameLozengeText.getWidth();

                this._hover.setDimension(textWidth, this._hover.getHeight());
                this._nameLozengeRectangle.setDimension(textWidth, this._nameLozengeRectangle.getHeight());
            },

            /**
             * Select the figure.
             */
            select: function () {
                this._super.apply(this, arguments);

                this._arrow.setBackgroundColor("#ffffff");
                this._arrow.setColor("#1a8cff");
                this._loopRectangle.setColor("#1a8cff");

                this._nameLozengeRectangle.setBackgroundColor("#333333");
                this._nameLozengeText.setFontColor("#ffffff");

            },

            /**
             * Set the figure's canvas.
             *
             * Also resizes the lozenges.
             */
            setCanvas: function () {
                this._super.apply(this, arguments);
                this._resizeLozenge();
                this._resizeNameLozenge();
            },

            /**
             * Set the figure's position.
             *
             * @param {number} x The figure's x position.
             * @param {number} y The figure's y position.
             */
            setPosition: function (x, y) {
                var boundingBox;
                this._super(x, y);
                boundingBox = this.getBoundingBox();
            },

            setValidation: function(validation) {
                this._validation = validation;
                if (!this._selected()) {
                    this._setTransitionLineColor();
                }
            },

            _selected: function () {
                return this._loopRectangle.getColor() === "#1a8cff";
            },

            _setTransitionLineColor: function () {
                var color = this._validation && this._validation.color|| "#cccccc";
                this._loopRectangle.setColor(color);
                this._arrow.setColor(color);
                this._arrow.setBackgroundColor(color);
            },

            /**
             * Deselect the figure.
             */
            unselect: function () {
                this._super.apply(this, arguments);

                this._arrow.setBackgroundColor("#cccccc");
                this._setTransitionLineColor();

                this._nameLozengeRectangle.setBackgroundColor("#ffffff");
                this._nameLozengeText.setFontColor("#333333");
            }
        });

});

AJS.namespace("JIRA.WorkflowDesigner.LoopedTransitionFigure", null, require("workflow-designer/looped-transition-figure"));