define("workflow-designer/looped-transition-container-view", [
    "workflow-designer/looped-transition-container-figure",
    "workflow-designer/looped-transition-figure",
    "workflow-designer/draw-2d-view",
    "workflow-designer/draw-2d",
    "workflow-designer/underscore"
], function(
    LoopedTransitionContainerFigure,
    LoopedTransitionFigure,
    Draw2DView,
    draw2d,
    _
) {
    return Draw2DView.extend(
    /** @lends JIRA.WorkflowDesigner.LoopedTransitionContainerView# */
    {
        initialize: function (options) {
            Draw2DView.prototype.initialize.apply(this, arguments);

            options = _.defaults({}, options, {
                immutable: false
            });

            this.immutable = options.immutable;
            this.workflowModel = options.workflowModel;
            this._setTransitionViews(options.transitionViews);
        },

        /**
         * Creates and configures the view's Draw2D figures.
         *
         * @private
         */
        _createAndConfigureFigure: function() {
            var figure = this._createFigure(),
                    instance = this;

            figure.onDrag = _.wrap(figure.onDrag, function (f) {
                f.apply(this, _.toArray(arguments).slice(1));
                instance.trigger("drag");
            });

            figure.onDragEnd = _.wrap(figure.onDragEnd, function (f) {
                f.apply(this, _.toArray(arguments).slice(1));
                instance._updatePosition();
                instance.trigger("dragEnd");
            });

            if (this.immutable) {
                figure.setDeleteable(false);
                figure.setResizeable(false);
            } else {
                figure.installEditPolicy(new draw2d.policy.figure.DragDropEditPolicy());
            }

            return figure;
        },

        _createFigure: function () {
            var figure = this._figure = new LoopedTransitionContainerFigure(),
                heightGap = 15,
                leftRightPadding = 15,
                maxTransitionWidth = 0,
                totalHeight = 0;

            figure.onDeselect = _.bind(this._onDeselect, this);
            figure.onSelect = _.bind(this._onSelect, this);
            figure.setSelectable(!this._immutable);

            this._transitionViews.each(function(transitionView) {
                transitionView.reset();
                var transitionFigure = transitionView.getFigure();
                figure.addTransitionFigure(transitionFigure);
                totalHeight += (transitionFigure.getHeight() + heightGap);
                maxTransitionWidth = Math.max(maxTransitionWidth, transitionFigure.getWidth());
            });

            figure.setDimension(maxTransitionWidth + (leftRightPadding * 2), totalHeight + heightGap);

            this._canvas.getLayer("looped-transitions-container").addFigure(figure);
            this._positionFigure();
            return figure;
        },

        getBoundingBox: function() {
            return this._figure.getBoundingBox();
        },

        getFigure: function () {
            return this._figure;
        },

        /**
         * Position the view's figure to match its model.
         *
         * @protected
         */
        _positionFigure: function () {
            if (this.model.hasCoordinates()) {
                this._figure.setPosition(this.model.get("x"), this.model.get("y"));
            }
        },

        /**
         * Render the container.
         *
         * @abstract
         * @return {JIRA.WorkflowDesigner.LoopedTransitionContainerView} `this`
         */
        render: function () {
            this._figure || (this._figure = this._createAndConfigureFigure());
            return this;
        },

        reset: function(transitionViews) {
            this._setTransitionViews(transitionViews);
            this.resetFigure();
        },

        resetFigure: function() {
            this._canvas.removeFigure(this._figure);
            this._figure = null;
            this.render();
        },

        /**
         * Set the container's position.
         *
         * @param {number} x The container's x position.
         * @param {number} y The container's y position.
         */
        setPosition: function (x, y) {
            this.model.set({
                x: x,
                y: y
            });
        },

        _setTransitionViews: function(transitionViews) {
            this._transitionViews = transitionViews;
            this.listenTo(this._transitionViews, "selected", this._transitionViewSelectedExternally);
        },

        _transitionViewSelectedExternally: function(selectedTransitionView) {
            this._selectedTransitionView &&
                this._selectedTransitionView !== selectedTransitionView &&
                this._selectedTransitionView.deselect();

            this._selectedTransitionView = selectedTransitionView;

            this._canvas.getSelection().getPrimary() !== this.getFigure() && this._canvas.getSelection().setPrimary(this.getFigure());
        },

        /**
         * Set the model's position in response to the rectangle moving.
         *
         * @private
         */
        _updatePosition: function () {
            var boundingBox = this.getBoundingBox();
            this.setPosition(boundingBox.getX(), boundingBox.getY());
        },

        _onDeselect: function() {
            this._selectedTransitionView && this._selectedTransitionView.deselect();
            this._selectedTransitionView = null;
        },

        _onSelect: function (selectionPath) {
            var selectedTransitionFigure,
                selectedTransitionView;
            selectedTransitionFigure = _.find(selectionPath, function(figure) {
                return (figure instanceof LoopedTransitionFigure);
            });
            if (selectedTransitionFigure) {
                selectedTransitionView = this._transitionViews.find(function(transitionView) {
                    return transitionView.getFigure() === selectedTransitionFigure;
                });
                if (selectedTransitionView) {
                    this._selectedTransitionView &&
                        this._selectedTransitionView !== selectedTransitionView &&
                        this._selectedTransitionView.deselect();

                    (!this._selectedTransitionView ||
                        this._selectedTransitionView !== selectedTransitionView) &&
                        selectedTransitionView.select();
                }
            } else {
                this._selectedTransitionView && this._selectedTransitionView.deselect();
            }
            this._selectedTransitionView = selectedTransitionView;
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.LoopedTransitionContainerView", null, require("workflow-designer/looped-transition-container-view"));