define("workflow-designer/global-transition-view", [
    "workflow-designer/dialogs/edit-global-transition-dialog-view",
    "workflow-designer/dialogs/delete-transition-dialog-view",
    "workflow-designer/global-transition-figure",
    "workflow-designer/inline-dialog-svg-positioner",
    "workflow-designer/selectable-figure-mixin",
    "workflow-designer/draw-2d-view",
    "aui/inline-dialog",
    "workflow-designer/templates",
    "workflow-designer/cocktail",
    "workflow-designer/underscore"
], function(
    EditGlobalTransitionDialogView,
    DeleteTransitionDialogView,
    GlobalTransitionFigure,
    InlineDialogSVGPositioner,
    SelectableFigureMixin,
    Draw2DView,
    InlineDialog,
    Templates,
    Cocktail,
    _
) {
    var GlobalTransitionView = Draw2DView.extend(
    /** @lends JIRA.WorkflowDesigner.GlobalTransitionView# */
    {
        /**
         * Initialise the global transition view.
         *
         * @classdesc The representation of a global transition on the designer canvas.
         * @constructs
         * @extends JIRA.WorkflowDesigner.Draw2DView
         * @param {object} options
         * @param {draw2d.Canvas} options.canvas The canvas to render to.
         * @param {boolean} [options.immutable=false] Whether the workflow is immutable.
         * @param {JIRA.WorkflowDesigner.StatusView} options.targetView The view of the transition's target status.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's workflow model.
         */
        initialize: function (options) {
            Draw2DView.prototype.initialize.apply(this, arguments);

            options = _.defaults({}, options, {
                immutable: false
            });

            this._immutable = options.immutable;
            this._targetView = options.targetView;
            this._workflowModel = options.workflowModel;

            this.listenTo(this._targetView, "move", this._onMove);
            this.listenTo(this._workflowModel, "validation:transitions", this._updateValidation);
        },

        _updateValidation: function(validationResult) {
            var validation = validationResult[this.model.get("id")];
            this._figure.setValidation(validation);
        },

        /**
         * Create the view's information dialog.
         *
         * @private
         */
        _createDialog: function () {
            var hoverElement = this._figure.getHoverElement(),
                id = _.uniqueId("global-transition-dialog-"),
                instance = this;

            function getArrowAttributes() {
                return {
                    fill : "#333333",
                    stroke : "#333333"
                };
            }

            function renderDialog(element, trigger, show) {
                var template = Templates.globalTransitionDialogContent;

                element.html(template({
                    name: instance.model.get("name"),
                    statusName: instance.model.get("target").get("name")
                }));

                show();
            }

            this._dialog = InlineDialog(hoverElement, id, renderDialog, {
                calculatePositions: InlineDialogSVGPositioner.calculatePosition,
                getArrowAttributes: getArrowAttributes,
                hideDelay: 50,
                onHover: true
            });
        },

        /**
         * Create the view's Draw2D figure.
         *
         * @private
         */
        _createFigure: function () {
            var figure;

            figure = this._figure = new GlobalTransitionFigure();
            figure.onDeselect = _.bind(this._onDeselect, this);
            figure.onSelect = _.bind(this._onSelect, this);
            figure.setSelectable(!this._immutable);
            this._immutable || (figure.onDoubleClick = _.bind(this.edit, this));

            this._canvas.getLayer("global-transitions").addFigure(figure);
            this._createDialog();
            this._positionFigure();
        },

        /**
         * Show a dialog that allows the user to delete the global transition.
         */
        destroy: function () {
            new DeleteTransitionDialogView({
                transitionModel: this.model,
                workflowModel: this._workflowModel
            }).show();
        },

        /**
         * Show a dialog for editing the global transition.
         */
        edit: function () {
            new EditGlobalTransitionDialogView({
                transitionModel: this.model,
                workflowModel: this._workflowModel
            }).show();
        },

        /**
         * Reposition the view in response to its target status moving.
         *
         * @private
         * @return {draw2d.Figure} The view's primary figure.
         * @see {@link JIRA.WorkflowDesigner.SelectableFigureMixin}
         */
        _getFigure: function () {
            return this._figure;
        },

        /**
         * Make the transition appear deselected and trigger a "deselected" event.
         *
         * @private
         * @see {@link JIRA.WorkflowDesigner.SelectableFigureMixin}
         */
        _onDeselect: function () {
            this._canvas.getLayer("global-transitions").addFigure(this._figure);
            this.trigger("deselected", this);
        },

        /**
         * Respond to the target view moving.
         *
         * @private
         */
        _onMove: function () {
            this._dialog && this._dialog.hide();
            this._positionFigure();
        },

        /**
         * Make the transition appear selected and trigger a "selected" event.
         *
         * @private
         * @see {@link JIRA.WorkflowDesigner.SelectableFigureMixin}
         */
        _onSelect: function () {
            this._canvas.getLayer("selected-transition").addFigure(this._figure);
            this.trigger("selected", this);
        },

        /**
         * Position the view's Draw2D figure.
         *
         * @private
         */
        _positionFigure: function() {
            var figureBoundingBox = this._figure.getBoundingBox(),
                targetBoundingBox = this._targetView.getBoundingBox();

            this._figure.setPosition(
                targetBoundingBox.getRight() + 2,
                targetBoundingBox.getCenter().getY() - figureBoundingBox.getHeight() / 2
            );
        },

        /**
         * Render the view.
         *
         * @return {JIRA.WorkflowDesigner.GlobalTransitionView} `this`
         */
        render: function () {
            this._figure || this._createFigure();
            return this;
        }
    });

    Cocktail.mixin(GlobalTransitionView, SelectableFigureMixin);

    return GlobalTransitionView;
});

AJS.namespace("JIRA.WorkflowDesigner.GlobalTransitionView", null, require("workflow-designer/global-transition-view"));