define("workflow-designer/global-transition-figure", [
    "workflow-designer/layout/locator/static-locator",
    "workflow-designer/triangle-figure",
    "workflow-designer/draw-2d",
    "workflow-designer/underscore",
    "jquery"
], function(
    StaticLocator,
    TriangleFigure,
    draw2d,
    _,
    jQuery
) {
    return draw2d.shape.basic.Rectangle.extend(
    /** @lends JIRA.WorkflowDesigner.GlobalTransitionFigure# */
    {
        /**
         * @constructs
         * @classdesc `GlobalTransitionView`'s Draw2D figure.
         * @extends draw2d.shape.basic.Rectangle
         */
        init: function () {
            var arrow, hover, line, lozenge, lozengeLocator;

            _.bindAll(this, "_onDoubleClick");
            this._super.apply(this, arguments);

            this.installEditPolicy(new draw2d.policy.figure.SelectionFeedbackPolicy());
            this.setAlpha(0);
            this.setDeleteable(false);
            this.setDimension(68, 14);
            this.setResizeable(false);

            // The line's points are set in setPosition().
            line = this._line = new draw2d.shape.basic.Line(0, 0, 0, 0);
            line.onDoubleClick = this._onDoubleClick;
            line.setColor("#cccccc");
            line.setStroke(1);
            this.addFigure(line, new draw2d.layout.locator.CenterLocator(this));

            arrow = this._arrow = new TriangleFigure();
            arrow.onDoubleClick = this._onDoubleClick;
            arrow.setDimension(7,8);
            arrow.setBackgroundColor("#cccccc");
            arrow.setColor("#cccccc");
            arrow.setStroke(1);
            this.addFigure(arrow, new StaticLocator(this, 0, 3));

            lozenge = this._createLozenge();
            lozengeLocator = new StaticLocator(this, 49, 0);
            this.addFigure(lozenge, lozengeLocator);

            hover = this._hover = new draw2d.shape.basic.Rectangle(19, 14);
            hover.onDoubleClick = this._onDoubleClick;
            hover.setAlpha(0);
            this.addFigure(hover, lozengeLocator);
        },

        /**
         * @return {draw2d.Figure} The "All" lozenge figure.
         * @private
         */
        _createLozenge: function () {
            var rectangle, text;

            rectangle = this._lozengeRectangle = new draw2d.shape.basic.Rectangle(19, 14);
            rectangle.setBackgroundColor("#333333");
            rectangle.setStroke(0);

            text = this._lozengeText = new draw2d.shape.basic.Label();
            text.setAlpha(0);
            text.setBold(true);
            text.setFontColor("#ffffff");
            text.setFontFamily("Arial");
            text.setFontSize(11);
            text.setStroke(0);
            text.setText(AJS.I18n.getText("common.words.all"));
            rectangle.addFigure(text, new draw2d.layout.locator.CenterLocator(rectangle));

            return rectangle;
        },

        /**
         * @return {element} The element that triggers a dialog on hover.
         */
        getHoverElement: function () {
            return this._hover.shape.node;
        },

        /**
         * @return {draw2d.util.ArrayList} The figure's ports.
         */
        getPorts: function () {
            // Draw2D's line figure doesn't implement `getPorts()` which causes
            // JS exceptions here. We don't have ports, so return an empty list.
            return new draw2d.util.ArrayList();
        },

        // Draw2D has a bug where only draggable figures are selectable. We
        // don't want global transitions to be draggable, thus this tomfoolery.
        onDrag: jQuery.noop,

        /**
         * Call the figure's `onDoubleClick` handler.
         *
         * @private
         */
        _onDoubleClick: function () {
            this.onDoubleClick.apply(this, arguments);
        },

        /**
         * Resize the "All" lozenge to ensure it contains its text.
         *
         * @private
         */
        _resizeLozenge: function () {
            var textWidth = this._lozengeText.getWidth();

            this._hover.setDimension(textWidth, this._hover.getHeight());
            this._lozengeRectangle.setDimension(textWidth, this._lozengeRectangle.getHeight());
        },

        /**
         * Select the figure.
         */
        select: function () {
            this._super.apply(this, arguments);

            this._arrow.setBackgroundColor("#ffffff");
            this._arrow.setColor("#1a8cff");
            this._line.setColor("#1a8cff");
        },

        /**
         * Set the figure's canvas.
         *
         * Also resizes the lozenge.
         */
        setCanvas: function () {
            this._super.apply(this, arguments);
            this._resizeLozenge();
        },

        /**
         * Set the figure's position.
         *
         * @param {number} x The figure's x position.
         * @param {number} y The figure's y position.
         */
        setPosition: function (x, y) {
            var boundingBox;

            this._super(x, y);

            // Line coordinates are absolute.
            boundingBox = this.getBoundingBox();
            this._line.setStartPoint(boundingBox.getLeft(), boundingBox.getCenter().getY());
            this._line.setEndPoint(boundingBox.getRight(), boundingBox.getCenter().getY());
        },

        setValidation: function(validation) {
            this._validation = validation;
            if (!this._isSelected()) {
                this._setLineColor();
            }
        },

        _isSelected: function() {
            return this._line.getColor() === "#1a8cff";
        },

        _setLineColor: function () {
            var color = this._validation && this._validation.color || "#cccccc";

            this._arrow.setBackgroundColor(color);
            this._arrow.setColor(color);
            this._line.setColor(color);
        },

        /**
         * Deselect the figure.
         */
        unselect: function () {
            this._super.apply(this, arguments);
            this._setLineColor();
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.GlobalTransitionFigure", null, require("workflow-designer/global-transition-figure"));