define("workflow-designer/status-view", [
    "workflow-designer/layout/locator/position-on-parent-bounds-locator",
    "workflow-designer/status-figure",
    "workflow-designer/dialogs/edit-status-dialog-view",
    "workflow-designer/dialogs/delete-status-dialog-view",
    "workflow-designer/direction",
    "workflow-designer/status-lozenge-colours",
    "workflow-designer/base-status-view",
    "workflow-designer/selectable-figure-mixin",
    "workflow-designer/cocktail",
    "workflow-designer/underscore",
    "workflow-designer/io/ajax/workflow-statuses-ajax-manager"
], function(
    PositionOnParentBoundsLocator,
    StatusFigure,
    EditStatusDialogView,
    DeleteStatusDialogView,
    Direction,
    StatusLozengeColours,
    BaseStatusView,
    SelectableFigureMixin,
    Cocktail,
    _,
    WorkflowStatusesAJAXManager
) {
    /**
     * @class JIRA.WorkflowDesigner.StatusView
     * @classdesc The representation of a non-initial workflow status on the designer canvas.
     * @mixes JIRA.WorkflowDesigner.SelectableFigureMixin
     * @extends JIRA.WorkflowDesigner.BaseStatusView
     */
    var StatusView = BaseStatusView.extend(
    /** @lends JIRA.WorkflowDesigner.StatusView# */
    {
        /**
         * Create and add a single port.
         *
         * @see {@link JIRA.WorkflowDesigner.Layout.Locator.PositionOnParentBoundsLocator}
         * @private
         */
        _addPort: function (bias, direction) {
            var instance = this,
                locator,
                port;

            port = this._createPort({
                connectionDirection: direction
            });

            locator = new PositionOnParentBoundsLocator({
                parent: this._figure,
                side: direction,
                bias: bias
            });

            this._figure.addPort(port, locator);

            port.bind("mouse:enter", function(e, previousHoverFigure) {
                instance._onMouseEnter(previousHoverFigure);
            });
            port.bind("mouse:leave", function(e, newHoverFigure) {
                instance._onMouseLeave(newHoverFigure);
            });

            port.bind("drag:start", _.bind(this._portStatusDragStart, this));
            port.bind("drag:end", _.bind(this._portStatusDragEnd, this));
            port.bind("select", _.bind(this.select, this));

            return port;
        },

        /**
         * Create the view's Draw2D figure.
         *
         * @private
         */
        _createFigure: function () {
            var figure, moveListener;

            moveListener = {
                onOtherFigureIsMoving: _.throttle(_.bind(this.trigger, this, "move"), 25)
            };

            figure = this._figure = new StatusFigure();
            figure.attachMoveListener(moveListener);
            figure.onDeselect = _.bind(this._onDeselect, this);
            figure.onMouseEnter = _.bind(this._onMouseEnter, this);
            figure.onMouseLeave = _.bind(this._onMouseLeave, this);
            figure.onSelect = _.bind(this._onSelect, this);
            figure.setSelectable(!this.immutable);

            this.listenTo(this.workflowModel, "validation:statuses", function(validationResult) {
                figure.setValidation(validationResult[this.model.get("stepId")]);
            });

            if (!this.immutable && this.workflowModel.get('permissions').get('editStatus')) {
                figure.onDoubleClick = _.bind(this.edit, this);
            }

            this._canvas.getLayer("statuses").addFigure(figure);
            this._createPorts();
            this._positionFigure();
            return figure;
        },

        /**
         * Create and add all the view's ports.
         *
         * @private
         */
        _createPorts: function() {
            this.ports = [
                // Top
                this._addPort(0.22, Direction.UP),
                this._addPort(0.5, Direction.UP),
                this._addPort(0.78, Direction.UP),

                // Right
                this._addPort(0.5, Direction.RIGHT),

                // Bottom
                this._addPort(0.78, Direction.DOWN),
                this._addPort(0.5, Direction.DOWN),
                this._addPort(0.22, Direction.DOWN),

                // Left
                this._addPort(0.5, Direction.LEFT)
            ];
        },

        /**
         * Destroy the status on the server after confirming the operation with the user.
         *
         * The {@link JIRA.WorkflowDesigner.StatusModel} is also destroyed, which causes the canvas to update.
         */
        destroy: function () {
            var onSuccess = function () {
                new DeleteStatusDialogView({
                    statusModel: this.model,
                    workflowModel: this.workflowModel
                }).show();
            }.bind(this);

            var onError = function (e) {
                new DeleteStatusDialogView({
                    errorMessage: e,
                    statusModel: this.model,
                    workflowModel: this.workflowModel
                }).show();
            }.bind(this);

            var request = WorkflowStatusesAJAXManager.validateRemoveStatus({
                statusId: this.model.get("statusId"),
                workflowName: this.workflowModel.get("name")
            });
            request.done(onSuccess).fail(onError);
        },

        /**
         * Show a dialog for editing the status.
         */
        edit: function () {
            new EditStatusDialogView({
                statusModel: this.model,
                workflowModel: this.workflowModel
            }).show();
        },

        /**
         * @returns {string} The status's background colour.
         * @private
         */
        _getBackgroundColor: function () {
            var colour,
                isSubtle = this._isSubtle(),
                statusCategory = this.model.get("statusCategory");

            if (statusCategory) {
                colour = StatusLozengeColours.getBackgroundColour({
                    colourName: statusCategory.colourName,
                    isSubtle: isSubtle
                });
            } else if (isSubtle) {
                colour = "#ffffff";
            } else {
                colour = "#4a6785";
            }

            return colour;
        },

        /**
         * @returns {string} The status's border colour.
         * @private
         */
        _getBorderColor: function () {
            var colour,
                isSubtle = this._isSubtle(),
                statusCategory = this.model.get("statusCategory");

            if (!isSubtle) {
                colour = "#ffffff";
            } else if (statusCategory) {
                colour = StatusLozengeColours.getBorderColour({
                    colourName: statusCategory.colourName,
                    isSubtle: isSubtle
                });
            } else {
                colour = "#4a6785";
            }

            return colour;
        },

        /**
         * @returns {draw2d.geo.Rectangle} The view's bounding box.
         */
        getBoundingBox: function () {
            return this._figure.getBoundingBox();
        },

        /**
         * @private
         * @return {draw2d.Figure} The view's primary Draw2D figure.
         * @see {@link JIRA.WorkflowDesigner.SelectableFigureMixin}
         */
        _getFigure: function () {
            return this._figure;
        },

        /**
         * @return {JIRA.WorkflowDesigner.StatusPort[]} The view's ports.
         */
        getPorts: function () {
            return this.ports;
        },

        /**
         * @return {string} The status's text colour.
         * @private
         */
        _getTextColor: function () {
            var colour,
                isSubtle = this._isSubtle(),
                statusCategory = this.model.get("statusCategory");

            if (statusCategory) {
                colour = StatusLozengeColours.getColour({
                    colourName: statusCategory.colourName,
                    isSubtle: isSubtle
                });
            } else if (isSubtle) {
                colour = "#4a6785";
            } else {
                colour = "#ffffff";
            }

            return colour;
        },

        /**
         * Highlight the status.
         */
        highlight: function() {
            this.immutable || _.invoke(this.getPorts(), "show");
            this.trigger("highlight");
        },

        /**
         * @returns {boolean} Whether the status should be rendered subtly.
         * @private
         */
        _isSubtle: function () {
            var hasCurrentStep = this.workflowModel.has("currentStepId"),
                isCurrent = this.model.get("stepId") === this.workflowModel.get("currentStepId"),
                isInitial = this.model.get("initial");

            return hasCurrentStep && !isCurrent && !isInitial;
        },

        /**
         * @param {draw2d.Figure} figure
         * @returns {boolean}
         * @private
         */
        _isThisStatusHighlightTarget: function (figure) {
            var isChild = figure && figure.getParent() === this._figure,
                isFigure = this._figure === figure,
                isPort = _.contains(this.ports, figure);

            return isChild || isFigure || isPort;
        },

        /**
         * Make the status appear deselected and trigger a "deselected" event.
         *
         * @private
         * @see {@link JIRA.WorkflowDesigner.SelectableFigureMixin}
         */
        _onDeselect: function () {
            this._canvas.getLayer("statuses").addFigure(this._figure);
            this.trigger("deselected", this);
            this.selected = false;
            this.unhighlight();
        },

        /**
         * @param {draw2d.Figure} previousHoverFigure
         * @private
         */
        _onMouseEnter: function (previousHoverFigure) {
            this._isThisStatusHighlightTarget(previousHoverFigure) || this.highlight();
        },

        /**
         * @param {draw2d.Figure} newHoverFigure
         * @private
         */
        _onMouseLeave: function (newHoverFigure) {
            this._isThisStatusHighlightTarget(newHoverFigure) || this.unhighlight();
        },

        /**
         * Make the status appear selected and trigger a "select" event.
         *
         * @private
         * @see {@link JIRA.WorkflowDesigner.SelectableFigureMixin}
         */
        _onSelect: function () {
            this._canvas.getLayer("selected-status").addFigure(this._figure);
            this.highlight();
            this.selected = true;
            this.trigger("selected", this);
        },

        /**
         * Triggers the event port:drag:end.
         *
         * @private
         */
        _portStatusDragEnd: function() {
            this.trigger("port:drag:end");
        },

        /**
         * Triggers the event port:drag:start.
         *
         * @private
         */
        _portStatusDragStart: function() {
            this.trigger("port:drag:start");
        },

        /**
         * Renders the status view
         *
         * @returns {JIRA.WorkflowDesigner.StatusView} this
         */
        render: function () {
            BaseStatusView.prototype.render.apply(this, arguments);
            this._figure.setBackgroundColor(this._getBackgroundColor());
            this._figure.setColor(this._getBorderColor());
            this._figure.setText(this.model.get("name"));
            this._figure.setTextColor(this._getTextColor());
            this._positionFigure();
            return this;
        },

        /**
         * Unhighlight the status.
         */
        unhighlight: function() {
            if (this.selected) {
                return;
            }

            this.immutable || _.invoke(this.getPorts(), "hide");
            this.trigger("unhighlight");
        },

        /**
         * @method
         */
        updatePortsVisibility: function() {
            _.invoke(this.getPorts(), this.selected ? "show" : "hide");
        }
    });

    Cocktail.mixin(StatusView, SelectableFigureMixin);

    return StatusView;
});

AJS.namespace("JIRA.WorkflowDesigner.StatusView", null, require("workflow-designer/status-view"));