AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:global-workflows",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var TestMocks = require("workflow-designer/test-mocks");
    var StatusModel = require("workflow-designer/status-model");
    var BaseStatusView = require("workflow-designer/base-status-view");

    module("BaseStatusView", {
        assertGetAngleToStatus: function (angle, sourceBounds, targetBounds) {
            var source = this.makeStatusView(sourceBounds),
                target = this.makeStatusView(targetBounds);

            equal(source.getAngleToStatus(target), angle, [
                "getAngleToStatus() returned ", angle, " for [", sourceBounds, "] to [", targetBounds, "]"
            ].join(""));
        },

        assertGetAngleToPort: function (angle, index, statusView) {
            equal(statusView.getAngleToPort(statusView.getPorts()[index]), angle,
                "getAngleToStatus() returned " + angle);
        },

        assertGetPortForAngle: function (angle, index, statusView) {
            var actual = statusView.getPortForAngle(angle),
                expected = statusView.getPorts()[index];

            ok(actual === expected, "getPortForAngle(" + angle + ") returned port " + index);
        },

        makeStatusView: function (bounds, ports) {
            var mockFigure = TestMocks.figure,
                statusModel = new StatusModel(),
                statusView;

            ports = _.map(ports, mockFigure);
            statusView = new BaseStatusView({model: statusModel});
            statusView._figure = mockFigure(bounds);
            statusView.getPorts = function () {
                return ports;
            };

            return statusView;
        },

        setup: function () {
            this.statusView = new BaseStatusView({
                model: new StatusModel()
            });
        }
    });

    test("getAngleToStatus()", function () {
        var tests = [
            [[0, 0, 1, 1], -135],
            [[1, 0, 1, 1], -90],
            [[2, 0, 1, 1], -45],
            [[2, 1, 1, 1], 0],
            [[2, 2, 1, 1], 45],
            [[1, 2, 1, 1], 90],
            [[0, 2, 1, 1], 135],
            [[0, 1, 1, 1], 180]
        ];

        _.each(tests, function (test) {
            this.assertGetAngleToStatus(test[1], [1, 1, 1, 1], test[0]);
        }, this);
    });

    test("getAngleToStatus() same position", function () {
        var bounds = [0, 0, 1, 1];
        this.assertGetAngleToStatus(0, bounds, bounds);
    });

    test("getAngleToPort()", function () {
        // A status view with a port every 45 degrees.
        var statusView = this.makeStatusView([0, 0, 5, 5], [
            [0, 0, 1, 1],
            [2, 0, 1, 1],
            [4, 0, 1, 1],
            [4, 2, 1, 1],
            [4, 4, 1, 1],
            [2, 4, 1, 1],
            [0, 4, 1, 1],
            [0, 2, 1, 1]
        ]);

        _.each(_.range(-135, 225, 45), function (angle, i) {
            this.assertGetAngleToPort(angle, i, statusView);
        }, this);
    });

    test("getPortForAngle()", function () {
        // A status view with a port every 45 degrees.
        var statusView = this.makeStatusView([0, 0, 5, 5], [
            [4, 2, 1, 1],
            [4, 4, 1, 1],
            [2, 4, 1, 1],
            [0, 4, 1, 1],
            [0, 2, 1, 1],
            [0, 0, 1, 1],
            [2, 0, 1, 1],
            [4, 0, 1, 1]
        ]);

        _.each(_.range(-360, 405, 45), function (angle, i) {
            this.assertGetPortForAngle(angle, i % statusView.getPorts().length, statusView);
        }, this);
    });

    test("getPortForAngle() over 180 degree boundary", function () {
        var statusView = this.makeStatusView([0, 0, 5, 5], [
            [0, 0, 1, 1],
            [2, 4, 1, 1]
        ]);

        this.assertGetPortForAngle(170, 0, statusView);
    });

    test("getPortForAngle() single port", function () {
        // A status view with a single port on the bottom.
        var statusView = this.makeStatusView([0, 0, 5, 5], [[2, 4, 1, 1]]);

        _.each(_.range(-180, 180, 45), function (angle) {
            this.assertGetPortForAngle(angle, 0, statusView);
        }, this);
    });

});
