define("workflow-designer/properties-panel/status-properties-view", [
    "workflow-designer/status-view",
    "workflow-designer/browser-utilities",
    "workflow-designer/messages",
    "workflow-designer/io/ajax/workflow-global-transitions-ajax-manager",
    "workflow-designer/properties-panel/properties-view",
    "workflow-designer/internal-api/panel-service",
    "workflow-designer/templates",
    "workflow-designer/underscore",
    "workflow-designer/analytics",
    "jquery"
], function(
    StatusView,
    BrowserUtilities,
    Messages,
    WorkflowGlobalTransitionsAJAXManager,
    PropertiesView,
    PanelService,
    Templates,
    _,
    Analytics,
    $
) {
    return PropertiesView.extend(
    /** @lends JIRA.WorkflowDesigner.PropertiesPanel.StatusPropertiesView# */
    {
        events: {
            "change #create-global-transition-sidebar": "_synchroniseGlobalTransition",
            "click .delete": "_destroy",
            "click .edit": "_edit",
            "click .add-transition-quick-fix": function() { this._openAddTransitionDialog(); }
        },

        template: Templates.PropertiesPanel.status,

        ui: {
            createGlobalTransition: "#create-global-transition-sidebar"
        },

        /**
         * Initialize the view.
         *
         * @classdesc The content of the properties panel when a status is selected.
         * @constructs
         * @extends JIRA.WorkflowDesigner.PropertiesPanel.PropertiesView
         * @param {object} options
         * @param {Backbone.View} options.selectedView The workflow designer's currently selected view.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The workflow model.
         */
        initialize: function () {
            _.bindAll(this, "_onCreateSuccess", "_onError", "_triggerTransactionEnd");

            PropertiesView.prototype.initialize.apply(this, arguments);
        },

        _openAddTransitionDialog: function() {
            $(".add-transition").click();
        },

        /**
         * Create a global transition to the status.
         *
         * Triggers a `transaction:end` event on completion.
         *
         * @private
         */
        _createGlobalTransition: function () {
            var options,
                request,
                statusModel = this._selectedView.model;

            options = {
                description: "",
                name: statusModel.get("name"),
                screenId: 0,
                statusId: statusModel.get("statusId"),
                workflowName: this._workflowModel.get("name")
            };

            request = WorkflowGlobalTransitionsAJAXManager.createGlobalTransition(options);
            request.always(this._triggerTransactionEnd).done(this._onCreateSuccess).fail(this._onError);
        },

        /**
         * Delete the global transition associated with the status.
         *
         * Triggers a `transaction:end` event on completion.
         *
         * @private
         */
        _deleteGlobalTransition: function () {
            var globalTransition = this._workflowModel.getGlobalTransitionForStatus(this._selectedView.model),
                instance = this,
                options,
                request;

            function removeTransition() {
                instance._workflowModel.removeTransition(globalTransition);
            }

            options = {
                transitionId: globalTransition.get("actionId"),
                workflowName: this._workflowModel.get("name")
            };

            request = WorkflowGlobalTransitionsAJAXManager.deleteGlobalTransition(options);
            request.always(this._triggerTransactionEnd).done(removeTransition).fail(this._onError);
        },

        /**
         * @private
         * @returns {boolean} Whether the selected status is the target of a global transition.
         */
        _hasGlobalTransition: function () {
            var statusModel = this._selectedView.model;
            return this._workflowModel.statusHasGlobalTransition(statusModel);
        },

        /**
         * Reset the `WorkflowModel` and re-select the status.
         *
         * Called when a global transition is successfully created.
         *
         * @param {object} layoutData The new workflow layout.
         * @private
         */
        _onCreateSuccess: function (layoutData) {
            var statusId = this._selectedView.model.id;

            this._workflowModel.reset(layoutData);
            this._canvasModel.set("selectedModel", this._workflowModel.getStatus(statusId));
        },

        /**
         * Reset the view and show an error message.
         *
         * Called when creating/destroying a global transition fails.
         *
         * @param {string} errorMessage The error message to show.
         * @private
         */
        _onError: function (errorMessage) {
            Messages.showErrorMessage(errorMessage);
            this.ui.createGlobalTransition.attr("checked", this._hasGlobalTransition());
        },

        /**
         * @return {object} The data to pass to the view's template.
         */
        serializeData: function () {
            var workflowName = this._workflowModel.get("name");
            var escapedWorkflowName = BrowserUtilities.escapeQueryParameter(workflowName);
            var statusId = this._selectedView.model.get("statusId");
            var stepId = this._selectedView.model.get("stepId");

            var panelContext = {
                workflowName: workflowName,
                stepId: stepId,
                statusId: statusId
            };

            return _.extend(PropertiesView.prototype.serializeData.apply(this), {
                editStatusPermission: this.options.workflowModel.get('permissions').get('editStatus'),
                deleteStatusPermission: this.options.workflowModel.get('permissions').get('deleteStatus'),
                hasGlobalTransition: this._hasGlobalTransition(),
                isDraft: this._workflowModel.get("draft"),
                workflowName: escapedWorkflowName,
                additionalPanels: PanelService.renderPanels("workflow.status.properties", panelContext)
            });
        },

        /**
         * Create or destroy a global transition to match the state of the checkbox.
         *
         * @private
         */
        _synchroniseGlobalTransition: function () {
            var isChecked;

            isChecked = this.ui.createGlobalTransition.is(":checked");
            isChecked ? this._createGlobalTransition() : this._deleteGlobalTransition();
            this.trigger("transaction:start");
        },

        /**
         * Trigger a `transaction:end` event.
         *
         * @private
         */
        _triggerTransactionEnd: function () {
            this.trigger("transaction:end");
        },

        _destroy: function () {
            Analytics.triggerRemoveStepClicked(this._workflowModel.get('permissions'), this._workflowModel.get("analyticsData"));
            PropertiesView.prototype._destroy.apply(this, arguments);
        }
    },

    /** @lends JIRA.WorkflowDesigner.PropertiesPanel.StatusPropertiesView */
    {
        /**
         * @param {Backbone.View} view View to check.
         * @return {boolean} Whether `view` is a {@link JIRA.WorkflowDesigner.StatusView}.
         */
        canHandleView: function (view) {
            return view instanceof StatusView;
        }
    }
    );
});

AJS.namespace("JIRA.WorkflowDesigner.PropertiesPanel.StatusPropertiesView", null, require("workflow-designer/properties-panel/status-properties-view"));
