define("workflow-designer/properties-panel/properties-panel-view", [
    "workflow-designer/properties-panel/status-properties-view",
    "workflow-designer/properties-panel/transition-properties-view",
    "workflow-designer/backbone",
    "workflow-designer/underscore",
    "jquery"
], function(
    StatusPropertiesView,
    TransitionPropertiesView,
    Backbone,
    _,
    jQuery
) {
    var PropertiesPanelView = Backbone.Marionette.ItemView.extend(
    /** @lends JIRA.WorkflowDesigner.PropertiesPanel.PropertiesPanelView# */
    {
        className: "properties-panel",

        events: {
            "aui-expander-collapsed": "_forceReflow",
            "aui-expander-expanded": "_forceReflow"
        },

        template: jQuery.noop,

        /**
         * Initialise the view.
         *
         * @classdesc The workflow designer's properties panel.
         * @constructs
         * @extends Backbone.View
         * @param {object} options
         * @param {JIRA.WorkflowDesigner.CanvasModel} options.canvasModel The application's `CanvasModel`.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's `WorkflowModel`.
         */
        initialize: function (options) {
            this._canvasModel = options.canvasModel;
            this._workflowModel = options.workflowModel;

            this.listenTo(this._canvasModel, "change:selectedView", this.render);
        },

        /**
         * Create a properties panel view for the currently selected object.
         *
         * @private
         * @returns {JIRA.WorkflowDesigner.PropertiesPanel.PropertiesPanelView|undefined} A `PropertiesPanelView`.
         */
        _createPropertiesView: function () {
            var propertiesView,
                triggerTransactionEnd = _.bind(this.trigger, this, "transaction:end"),
                triggerTransactionStart = _.bind(this.trigger, this, "transaction:start");

            propertiesView = PropertiesPanelView.create({
                canvasModel: this._canvasModel,
                selectedView: this._canvasModel.get("selectedView"),
                workflowModel: this._workflowModel
            });

            propertiesView && this.listenTo(propertiesView, {
                "transaction:end": triggerTransactionEnd,
                "transaction:start": triggerTransactionStart
            });

            return propertiesView;
        },

        /**
         * Remove the current properties view before rendering.
         */
        onBeforeRender: function () {
            this._propertiesView && this._propertiesView.remove();
        },

        /**
         * Create and render a properties view.
         */
        onRender: function () {
            this._propertiesView = this._createPropertiesView();

            if (this._propertiesView) {
                this._propertiesView.render().$el.appendTo(this.el);

                // CAS-638: Safari does not trigger a reflow when the panel's content is changed. We trigger a reflow
                // by calling `height`. Also, this ensures that `this._propertiesView.$el.height !== 0` in Safari
                // so that applyDescriptionExpander actually works (it needs a correct height).
                this.$el.height();
                this._propertiesView.applyDescriptionExpander();
            }
        },

        /**
         * Force a reflow of the properties panel.
         *
         * This is used to work around an IE bug where the panel's maximum
         * height is not enforced after collapsing/expanding the description.
         *
         * @private
         */
        _forceReflow: function () {
            var setHeight = _.bind(this.$el.css, this.$el, "height");

            setHeight("1px");
            _.defer(setHeight, "");
        }
    },

    /** @lends JIRA.WorkflowDesigner.PropertiesPanel.PropertiesPanelView */
    {
        /**
         * Create a properties view for the currently selected object.
         *
         * @param {object} options Options to pass to the view's constructor.
         * @param {JIRA.WorkflowDesigner.CanvasModel} options.canvasModel The canvas model.
         * @returns {JIRA.WorkflowDesigner.PropertiesPanel.PropertiesView|undefined} A properties view.
         */
        create: function (options) {
            var selectedView = options.canvasModel.get("selectedView"),
                viewClasses,
                viewClass;

            // We can't define viewClasses as a private variable inside our IIFE,
            // because the subclasses might not be present when this class loads.
            viewClasses = [
                StatusPropertiesView,
                TransitionPropertiesView
            ];

            viewClass = _.find(viewClasses, function (viewClass) {
                return viewClass.canHandleView(selectedView);
            });

            return viewClass && new viewClass(options);
        }
    });

    return PropertiesPanelView;
});

AJS.namespace("JIRA.WorkflowDesigner.PropertiesPanel.PropertiesPanelView", null, require("workflow-designer/properties-panel/properties-panel-view"));