AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var PropertiesView = require("workflow-designer/properties-panel/properties-view");
    var TransitionPropertiesView = require("workflow-designer/properties-panel/transition-properties-view");
    var StatusPropertiesView = require("workflow-designer/properties-panel/status-properties-view");
    var PropertiesPanelView = require("workflow-designer/properties-panel/properties-panel-view");
    var jQuery = require("jquery");
    var WorkflowModel = require("workflow-designer/workflow-model");
    var TransitionModel = require("workflow-designer/transition-model");
    var TransitionView = require("workflow-designer/transition-view");
    var CanvasModel = require("workflow-designer/canvas-model");
    var StatusModel = require("workflow-designer/status-model");
    var StatusView = require("workflow-designer/status-view");
    var GlobalTransitionView = require("workflow-designer/global-transition-view");
    var TestUtilities = require("workflow-designer/test-utilities");

    module("PropertiesPanelView", {
        /**
         * Change the workflow model's selected view and assert that a given properties view is shown.
         *
         * @param {JIRA.WorkflowDesigner.PropertiesPanel.PropertiesView} selectedView The view to select.
         * @param {object} viewContainer The object containing the properties view.
         * @param {string} viewName The name of the properties view.
         */
        assertPropertiesView: function (selectedView, view) {
            var viewSpy = view.prototype.initialize;
            var renderSpy = view.prototype.render;

            this.canvasModel.set("selectedView", selectedView);
            equal(viewSpy.callCount, 1, "A property view instance was created");
            equal(renderSpy.callCount, 1, "The view's render method was called");
            ok(this.propertiesPanelView.$el.has(renderSpy.returnValues[0].el).length,
                "The view's element was added to the properties panel");
        },

        /**
         * Create and return a properties panel view.
         *
         * @return {PropertiesPanelView} The view.
         */
        createPropertiesPanelView: function () {
            this.workflowModel = new WorkflowModel();
            this.canvasModel = new CanvasModel({}, {
                workflowModel: this.workflowModel
            });
            this.canvasView = TestUtilities.testCanvasView({
                canvasModel: this.canvasModel,
                workflowModel: this.workflowModel
            });
            this.canvas = this.canvasView.canvas;

            return new PropertiesPanelView({
                canvasModel: this.canvasModel,
                container: this.container,
                workflowModel: this.workflowModel
            });
        },

        setup: function () {
            this.sandbox = sinon.sandbox.create();

            this.sandbox.spy(PropertiesView.prototype, "initialize");
            this.sandbox.spy(PropertiesPanelView.prototype, "initialize");
            this.sandbox.spy(StatusPropertiesView.prototype, "initialize");
            this.sandbox.spy(PropertiesView.prototype, "render");
            this.propertiesPanelViewRenderSpy = this.sandbox.spy(PropertiesPanelView.prototype, "render");

            this.container = jQuery("#qunit-fixture");
            this.propertiesPanelView = this.createPropertiesPanelView();
        },

        teardown: function () {
            this.sandbox.restore();
        }
    });

    test("A TransitionPropertiesView is shown when a global transition is selected", function () {
        TestUtilities.fakeTimer(function () {
            var targetView = this.canvasView.addStatus(this.workflowModel.addStatus({})),
                transitionView;

            transitionView = new GlobalTransitionView({
                canvas: this.canvas,
                model: new TransitionModel({
                    source: new StatusModel({stepId: 1})
                }),
                targetView: targetView,
                workflowModel: new WorkflowModel()
            });

            this.assertPropertiesView(transitionView, TransitionPropertiesView);
        }, this);
    });

    test("A TransitionPropertiesView is shown when a transition is selected", function () {
        var sourceView = this.canvasView.addStatus(this.workflowModel.addStatus({})),
            targetView = this.canvasView.addStatus(this.workflowModel.addStatus({})),
            transitionView,
            workflowModel = new WorkflowModel();

        transitionView = new TransitionView({
            canvas: this.canvas,
            canvasModel: new CanvasModel({}, {
                workflowModel: workflowModel
            }),
            model: new TransitionModel({
                source: new StatusModel({stepId: 1})
            }),
            sourceView: sourceView,
            targetView: targetView,
            workflowModel: workflowModel
        });

        this.assertPropertiesView(transitionView, TransitionPropertiesView);
    });

    test("No PropertiesView is shown when nothing is selected", function () {
        var initializeSpy;

        // Use a status view as our "initial" state
        this.canvasModel.set("selectedView", new StatusView({
            canvas: this.canvas,
            model: new StatusModel({}),
            workflowModel: new WorkflowModel()
        }));

        initializeSpy = PropertiesView.prototype.initialize;
        initializeSpy.reset();
        this.canvasModel.set("selectedView", null);
        equal(initializeSpy.callCount, 0, "No PropertiesView was created");
    });

    test("A StatusPropertiesView is shown when a status selected", function () {
        var statusView = new StatusView({
            canvas: this.canvas,
            model: new StatusModel({}),
            workflowModel: new WorkflowModel()
        });

        this.assertPropertiesView(statusView, StatusPropertiesView);
    });

    test("The view renders when the selected view changes", function () {
        this.canvasModel.set("selectedView", {});
        equal(this.propertiesPanelViewRenderSpy.callCount, 1, "render() was called");
    });

    test("It can create subclasses based on the selected view", function () {
        var canvas,
            create,
            globalTransitionView,
            statusView,
            transitionView;

        canvas = TestUtilities.testDraw2DCanvas();
        statusView = new StatusView({
            canvas: canvas,
            model: new StatusModel(),
            workflowModel: this.workflowModel
        });
        transitionView = new TransitionView({
            canvas: TestUtilities.testDraw2DCanvas(),
            canvasModel: new CanvasModel({}, {
                workflowModel: this.workflowModel
            }),
            immutable: false,
            model: new TransitionModel(),
            sourceView: statusView,
            targetView: statusView,
            workflowModel: this.workflowModel
        });
        globalTransitionView = new GlobalTransitionView({
            canvas: TestUtilities.testDraw2DCanvas(),
            immutable: true,
            workflowModel: this.workflowModel,
            targetView: statusView
        });
        create = PropertiesPanelView.create;

        this.canvasModel.set("selectedView", transitionView, {silent: true});
        ok(create({canvasModel: this.canvasModel}) instanceof TransitionPropertiesView,
            "Can handle a transitions view");

        this.canvasModel.set("selectedView", globalTransitionView, {silent: true});
        ok(create({canvasModel: this.canvasModel}) instanceof TransitionPropertiesView,
            "Can handle a global transition view");

        this.canvasModel.set("selectedView", statusView, {silent: true});
        ok(create({canvasModel: this.canvasModel}) instanceof StatusPropertiesView,
            "Can handle a status view");
    });

    test("transaction:start/end events are forwarded from PropertiesViews", function () {
        var propertiesPanelView,
            propertiesView,
            statusPropertiesViewSpy = StatusPropertiesView.prototype.initialize,
            transactionEndSpy = sinon.spy(),
            transactionStartSpy = sinon.spy();

        propertiesPanelView = this.createPropertiesPanelView().render();
        this.canvasModel.set("selectedView", new StatusView({
            canvas: this.canvas,
            model: new StatusModel({}),
            workflowModel: this.workflowModel
        }));

        propertiesPanelView.on({
            "transaction:end": transactionEndSpy,
            "transaction:start": transactionStartSpy
        });

        propertiesView = statusPropertiesViewSpy.thisValues[0];
        propertiesView.trigger("transaction:start");
        equal(transactionStartSpy.callCount, 1, "The transaction:start event was forwarded");

        propertiesView.trigger("transaction:end");
        equal(transactionStartSpy.callCount, 1, "The transaction:end event was forwarded");
    });

});
