define("workflow-designer/reuse-transition-form-view", [
    "workflow-designer/io/ajax/workflow-transitions-ajax-manager",
    "workflow-designer/add-transition-form-view",
    "jira/ajs/select/single-select",
    "workflow-designer/analytics",
    "workflow-designer/templates",
    "workflow-designer/backbone",
    "workflow-designer/underscore",
    "jquery"
], function(
    WorkflowTransitionsAJAXManager,
    AddTransitionFormView,
    SingleSelect,
    Analytics,
    Templates,
    Backbone,
    _,
    jQuery
) {
    return AddTransitionFormView.extend({
        events: {
            "simpleClick .create-new-transition": "_triggerCreateNewTransition"
        },

        template: Templates.reuseTransitionForm,

        ui: {
            noTransitionsMessage: ".no-transitions-message",
            sourceStepId: "#transition-source-step-id",
            targetStepId: "#transition-target-step-id",
            transitionId: "#transition-id"
        },

        /**
         * Initialize the view.
         *
         * @constructs
         * @classdesc A form for reusing transitions (creating common transitions).
         *
         * This view must be shown in a `Marionette.Region` to work correctly.
         * @extends Backbone.Marionette.ItemView
         * @param {object} options
         * @param {JIRA.WorkflowDesigner.TransitionModel} options.model The new transition model.
         * @param {object[]} options.sourceStatuses The source statuses that can be selected.
         * @param {object[]} options.targetStatuses The target statuses that can be selected.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's workflow model.
         */
        initialize: function () {
            _.bindAll(this, "_onSuccess", "_refreshTransitionOptions");
        },

        /**
         * @returns {JIRA.WorkflowDesigner.TransitionModel} The currently selected transition.
         * @private
         */
        _getSelectedTransition: function () {
            var actionId = Number(this.ui.transitionId.find(":selected").val());
            return this.options.workflowModel.get("transitions").findWhere({
                actionId: actionId
            });
        },

        /**
         * @returns {JIRA.WorkflowDesigner.TransitionModel[]} Transitions that can be reused.
         * @private
         */
        _getTransitions: function () {
            var selectedTargetStatus = this._getSelectedTargetStatus(),
                transitions = this.options.workflowModel.get("transitions");

            function getActionId(transition) {
                return transition.get("actionId");
            }

            function getName(transition) {
                return transition.get("name").toLowerCase();
            }

            function hasSelectedTargetStatus(transition) {
                return transition.get("target") === selectedTargetStatus;
            }

            function isInitial(transition) {
                return transition.get("source") && transition.get("source").isInitial();
            }

            return transitions
                .chain()
                .filter(hasSelectedTargetStatus)
                .reject(isInitial)
                .sortBy(getName)
                .uniq(getActionId)
                .value();
        },

        onShow: function () {
            AddTransitionFormView.prototype.onShow.apply(this, arguments);

            this._transitionField = new SingleSelect({
                element: this.ui.transitionId,
                revertOnInvalid: true
            });

            this._refreshTransitionOptions();
            this.ui.targetStepId.on("selected", this._refreshTransitionOptions);
        },

        _onSuccess: function () {
            var attributes;

            attributes = this._getSelectedTransition().omit("id", "sourceAngle", "targetAngle");
            attributes = _.extend(attributes, {
                source: this._getSelectedSourceStatus()
            });

            Analytics.triggerAddCommonTransition(this.options.workflowModel.get("permissions"), this.options.workflowModel.get("analyticsData"));
            this.model.set(attributes);
        },

        /**
         * Update the transition field's options or hide it if there are none.
         *
         * @private
         */
        _refreshTransitionOptions: function () {
            var options;

            options = _.map(this._getTransitions(), function (transition, index) {
                var option;

                option = jQuery("<option>").attr({
                    value: transition.get("actionId")
                }).text(transition.get("name"));

                index === 0 && option.attr("selected", "selected");
                return option[0];
            });

            this._transitionField.$container.toggle(options.length > 0);
            this._transitionField.model.$element.html(options).trigger("reset");
            this.ui.noTransitionsMessage.toggle(options.length < 1);
        },

        serializeData: function () {
            return {
                sourceStatuses: this.options.sourceStatuses,
                targetStatuses: this.options.targetStatuses
            };
        },

        /**
         * Attempt to create a common transition.
         *
         * @returns {jQuery.Deferred} A promise that is resolved on completion.
         */
        submit: function () {
            var deferred = jQuery.Deferred(),
                hasNoTransitions = !this._getTransitions().length,
                options,
                request;

            if (hasNoTransitions) {
                return deferred.reject(AJS.I18n.getText("workflow.designer.no.transitions.to.this.status")).promise();
            }

            options = {
                sourceStatusId: this._getSelectedSourceStatus().get("statusId"),
                transitionId: this._getSelectedTransition().get("actionId"),
                workflowName: this.options.workflowModel.get("name")
            };

            request = WorkflowTransitionsAJAXManager.addCommonTransition(options);
            request.done(this._onSuccess).done(deferred.resolve).fail(deferred.reject);
            return deferred.promise();
        },

        /**
         * Trigger a "createNewTransition" event.
         *
         * @param {object} [e] The event that caused this method to be called.
         * @private
         */
        _triggerCreateNewTransition: function (e) {
            e && e.preventDefault();
            this.trigger("createNewTransition");
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.ReuseTransitionFormView", null, require("workflow-designer/reuse-transition-form-view"));
