define("workflow-designer/new-transition-form-view", [
    "workflow-designer/add-transition-form-view",
    "workflow-designer/io/ajax/workflow-looped-transitions-ajax-manager",
    "workflow-designer/io/ajax/workflow-global-transitions-ajax-manager",
    "workflow-designer/io/ajax/workflow-transitions-ajax-manager",
    "jira/ajs/select/single-select",
    "workflow-designer/templates",
    "workflow-designer/backbone",
    "workflow-designer/underscore",
    "jquery"
], function(
    AddTransitionFormView,
    WorkflowLoopedTransitionsAJAXManager,
    WorkflowGlobalTransitionsAJAXManager,
    WorkflowTransitionsAJAXManager,
    SingleSelect,
    Templates,
    Backbone,
    _,
    jQuery
) {
    var AnyStatusStepId = -65536,
        ToItselfStepId = -65537;

    return AddTransitionFormView.extend({

        template: Templates.newTransitionForm,

        ui: {
            description: "#transition-description",
            name: "#transition-name",
            screenId: "#transition-screen-id",
            sourceStepId: "#transition-source-step-id",
            targetStepId: "#transition-target-step-id"
        },

        /**
         * Initialize the view.
         *
         * @constructs
         * @classdesc A form for creating new transitions.
         *
         * This view must be shown in a `Marionette.Region` to work correctly.
         * @extends JIRA.WorkflowDesigner.AddTransitionFormView
         * @param {object} options
         * @param {object[]} options.screens The screens that can be selected.
         * @param {object[]} options.sourceStatuses The source statuses that can be selected.
         * @param {object[]} options.targetStatuses The target statuses that can be selected.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's workflow model.
         */
        initialize: function () {
            _.bindAll(this, "_onAddTransitionSuccess");

            function makeOption(name, value) {
                return {
                    text: name,
                    value: value,
                    selected: false
                };
            }

            this.options = jQuery.extend(true, {}, this.options, {});
            this.options.sourceStatuses.push(
                makeOption(AJS.I18n.getText("workflow.designer.status.any.status"), AnyStatusStepId));
            this.options.targetStatuses.push(
                makeOption(AJS.I18n.getText("workflow.designer.status.to.itself"), ToItselfStepId));
        },

        /**
         * @returns {number} The ID of the selected screen.
         * @private
         */
        _getSelectedScreenId: function () {
            return Number(this.ui.screenId.find(":selected").val() || 0);
        },

        /**
         * @returns {string} The name of the selected screen.
         * @private
         */
        _getSelectedScreenName: function () {
            return this.ui.screenId.find(":selected").text();
        },

        onShow: function () {
            AddTransitionFormView.prototype.onShow.apply(this, arguments);

            new SingleSelect({
                element: this.ui.screenId,
                revertOnInvalid: true
            });
        },

        /**
         * @param {object} response An AJAX response.
         * @private
         */
        _onAddTransitionSuccess: function (response) {
            var attributes;
            attributes = _.extend({}, response, {
                description: this.ui.description.val(),
                globalTransition: response.global,
                loopedTransition: response.looped,
                name: this.ui.name.val(),
                screenId: this._getSelectedScreenId(),
                screenName: this._getSelectedScreenName(),
                source: this._getSelectedSourceStatus(),
                target: this._getSelectedTargetStatus()
            });

            this.model.set(attributes);
        },

        serializeData: function () {
            return {
                screens: this.options.screens,
                screensPermission: this.options.workflowModel.get('permissions').get('selectScreenOnTransition'),
                sourceStatuses: this.options.sourceStatuses,
                targetStatuses: this.options.targetStatuses
            };
        },

        /**
         * Should we use the global transition path?
         * @private
         */
        _shouldUseGlobalTransition: function (options) {
            return options.sourceStepId === AnyStatusStepId && options.targetStepId !== ToItselfStepId;
        },

        /**
         * Should we use the looped transition path?
         * @private
         */
        _shouldUseLoopedTransition: function (options) {
            return options.sourceStepId === AnyStatusStepId && options.targetStepId === ToItselfStepId;
        },

        /**
         * Should we use a normal transition loop?
         * @private
         */
        _shouldUseNormalTransitionLoop: function (options) {
            return options.sourceStepId !== AnyStatusStepId && options.targetStepId === ToItselfStepId;
        },

        /**
         * Set up the adding a looped transition request path.
         * @private
         */
        _setupLoopedTransition: function (options) {
            var request = WorkflowLoopedTransitionsAJAXManager.addLoopedTransition(options);
            return request;
        },

        /**
         * Set up the add global transition request path.
         * @private
         */
        _setupGlobalTransition: function (options) {
            var request;
            options = _.extend(options, {
                statusId: this._getSelectedTargetStatus().get("statusId")
            });

            request = WorkflowGlobalTransitionsAJAXManager.addGlobalTransition(options);
            return request;
        },

        /**
         * Set up the adding a normal transition request path.
         * @private
         */
        _setupNormalTransition: function (options) {
            var request;
            options = _.extend(options, {
                sourceStepId: this._getSelectedSourceStatus().get("stepId"),
                targetStepId: this._getSelectedTargetStatus().get("stepId")
            });

            request = WorkflowTransitionsAJAXManager.addTransition(options);
            return request;
        },

        /**
         * Set up the adding a normal transition loop request path.
         * @private
         */
        _setupNormalTransitionLoop: function (options) {
            var request;
            options = _.extend(options, {
                sourceStepId: this._getSelectedSourceStatus().get("stepId"),
                targetStepId: this._getSelectedSourceStatus().get("stepId")
            });

            this.ui.targetStepId.val(this.ui.sourceStepId.val());
            request = WorkflowTransitionsAJAXManager.addTransition(options);
            return request;
        },

        /**
         * Attempt to create a new transition.
         *
         * @returns {jQuery.Deferred} A promise that is resolved on completion.
         */
        submit: function () {
            var deferred = jQuery.Deferred(),
                options,
                request;

            options = {
                description: this.ui.description.val(),
                name: this.ui.name.val(),
                screenId: this._getSelectedScreenId(),
                sourceStepId: Number(this.ui.sourceStepId.find(":selected").val()),
                targetStepId: Number(this.ui.targetStepId.find(":selected").val()),
                workflowName: this.options.workflowModel.get("name")
            };

            if (this._shouldUseLoopedTransition(options)) {
                request = this._setupLoopedTransition(options);
            } else if (this._shouldUseGlobalTransition(options)) {
                request = this._setupGlobalTransition(options);
            } else if (this._shouldUseNormalTransitionLoop(options)) {
                request = this._setupNormalTransitionLoop(options);
            } else {
                request = this._setupNormalTransition(options);
            }

            request.done(this._onAddTransitionSuccess).done(deferred.resolve).fail(deferred.reject);
            return deferred.promise();
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.NewTransitionFormView", null, require("workflow-designer/new-transition-form-view"));