define("workflow-designer/dialogs/transition-dialog-view", [
    "workflow-designer/dialogs/form-dialog-view",
    "workflow-designer/io/ajax/screens-ajax-manager",
    "workflow-designer/messages",
    "workflow-designer/backbone",
    "workflow-designer/underscore",
    "jquery"
], function(
    FormDialogView,
    ScreensAJAXManager,
    Messages,
    Backbone,
    _,
    jQuery
) {
    return FormDialogView.extend(
    /** @lends JIRA.WorkflowDesigner.Dialogs.TransitionDialogView# */
    {
        /**
         * Initialise the view.
         *
         * @constructs
         * @classdesc
         * A dialog for creating/editing a transition.
         *
         * This view is abstract; do not create instances of it, use those that extend from it.
         * @extends JIRA.WorkflowDesigner.Dialogs.FormDialogView
         * @param {object} options
         * @param {JIRA.WorkflowDesigner.CanvasModel} options.canvasModel The application's `CanvasModel`.
         * @param {JIRA.WorkflowDesigner.TransitionModel} options.transitionModel The transition being created.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's `WorkflowModel`.
         */
        initialize: function (options) {
            _.bindAll(this, "_onError", "_onLoadError", "_onSubmit", "_onSuccess");

            this._canvasModel = options.canvasModel;
            this._transitionModel = options.transitionModel;
            this._workflowModel = options.workflowModel;
        },

        /**
         * @returns {jQuery.Deferred} A deferred that is resolved with screen options.
         * @protected
         */
        _getScreenOptions: function () {
            function makeOption(object) {
                return {
                    text: object.name,
                    value: object.id,
                    selected: object.id === selectedScreenId
                };
            }
            
            var deferred = jQuery.Deferred();
            if (this._hasScreenPermissions()) {
                var request = ScreensAJAXManager.getScreens(),
                    selectedScreenId = this._transitionModel.get("screenId");

                request.done(function (screens) {
                    deferred.resolve(_.map(screens, makeOption));
                });

                request.fail(deferred.reject);
                return deferred.promise();
            } else {
                deferred.resolve([{id: 0, name: AJS.I18n.getText("workflow.designer.no.screen.for.transition")}]);
                return deferred.promise();
            }
        },

        _hasScreenPermissions: function () {
            return !!this._workflowModel.get('permissions').get('selectScreenOnTransition');
        },
        
        /**
         * Display an error message and hide the dialog.
         *
         * Called when loading of the dialog's content fails.
         *
         * @param {string} errorMessage The error message to show.
         * @protected
         */
        _onLoadError: function (errorMessage) {
            Messages.showErrorMessage(errorMessage);
            this.hide();
        },

        /**
         * @param {jQuery.Event} e Submit event
         * @private
         */
        _onSubmit: function (e) {
            e.preventDefault();
            this.disable();
            this.removeErrorMessages();
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.Dialogs.TransitionDialogView", null, require("workflow-designer/dialogs/transition-dialog-view"));