define("workflow-designer/dialogs/status-dialog-view", [
    "workflow-designer/dialogs/form-dialog-view",
    "workflow-designer/io/ajax/status-categories-ajax-manager",
    "workflow-designer/messages",
    "workflow-designer/backbone",
    "workflow-designer/underscore",
    "jquery"
], function(
    FormDialogView,
    StatusCategoriesAJAXManager,
    Messages,
    Backbone,
    _,
    jQuery
) {
    var StatusCategorySingleSelect = JIRA.StatusCategorySingleSelect;

    var undefinedStatusCategoryId = 1;

    return FormDialogView.extend (
    /** @lends JIRA.WorkflowDesigner.Dialogs.StatusDialogView# */
    {
        ui: {
            description: "#status-description",
            name: "#status-name",
            statusCategoryId: "#status-category-id"
        },

        /**
         * Initialise the dialog.
         *
         * @constructs
         * @classdesc The workflow designer's "Status" dialog.
         *
         * This view is abstract; do not create instances of it, use those that extend from it.
         * @extends JIRA.WorkflowDesigner.Dialogs.FormDialogView
         * @param {object} options
         * @param {JIRA.WorkflowDesigner.StatusModel} options.statusModel The application's `StatusModel`.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's `WorkflowModel`.
         */
        initialize: function (options) {
            _.bindAll(this, "_onSubmit", "_onSuccess", "_onError");

            this._workflowModel = options.workflowModel;
            this._statusModel = options.statusModel;
        },

        /**
         * @returns {function} A function that retrieves the dialog's content.
         * @private
         */
        _getContent: function () {
            var instance = this;

            return function (callback) {
                var data = instance.serializeData(),
                    request = instance._getStatusCategoryOptions();

                request.done(function (statusCategoryOptions) {
                    _.extend(data, {
                        showStatusCategoryField: true,
                        statusCategoryOptions: statusCategoryOptions
                    });

                    callback(instance.template(data));
                });

                request.fail(function (errorMessage) {
                    Messages.showErrorMessage(errorMessage);
                    instance.hide();
                });
            };
        },

        /**
         * @returns {number|undefined} The ID of the status's current category or `undefined`.
         * @private
         */
        _getCurrentStatusCategoryId: function () {
            var currentStatusCategory = this._statusModel.get("statusCategory");
            return currentStatusCategory && currentStatusCategory.id;
        },

        /**
         * @private
         * @returns {jQuery.Deferred} A deferred that is resolved with all
         *         status categories that should be shown in the UI.
         */
        _getStatusCategories: function () {
            var deferred = jQuery.Deferred(),
                request = StatusCategoriesAJAXManager.getStatusCategories();

            function isUndefinedStatusCategory(statusCategory) {
                return statusCategory.id === undefinedStatusCategoryId;
            }

            request.done(function (statusCategories) {
                deferred.resolve(_.reject(statusCategories, isUndefinedStatusCategory));
            });

            request.fail(deferred.reject);
            return deferred;
        },

        /**
         * @private
         * @returns {jQuery.Deferred} A deferred that is resolved with the
         *         status category options that should be shown.
         */
        _getStatusCategoryOptions: function () {
            var currentStatusCategoryId = this._getCurrentStatusCategoryId(),
                deferred = jQuery.Deferred(),
                request = this._getStatusCategories();

            function makeOption(statusCategory) {
                return {
                    colorName: statusCategory.colorName,
                    key: statusCategory.key,
                    selected: statusCategory.id === currentStatusCategoryId,
                    text: statusCategory.name,
                    value: statusCategory.id
                };
            }

            request.done(function (statusCategories) {
                var options;

                options = _.map(statusCategories, makeOption);
                options.push({
                    isPlaceholder: true,
                    selected: currentStatusCategoryId === undefinedStatusCategoryId,
                    text: AJS.I18n.getText("admin.issuesettings.statuses.status.category.please.select")
                });

                deferred.resolve(options);
            });

            request.fail(deferred.reject);
            return deferred;
        },

        /**
         * Enhance the status category field if it's present.
         *
         * @private
         */
        _onContentRefresh: function () {
            if (this.ui.statusCategoryId.length) {
                new StatusCategorySingleSelect({
                    element: this.ui.statusCategoryId,
                    revertOnInvalid: true
                });
            }
        },

        /**
         * @param {jQuery.Event} e Submit event
         * @param {function} callback Function to call when this event has been handled
         * @private
         */
        _onSubmit: function (e, callback) {
            var options, request;

            e.preventDefault();
            this.disable();
            this.removeErrorMessages();

            options = {
                description: this.ui.description.val(),
                name: this.ui.name.val(),
                statusCategoryId: this.ui.statusCategoryId.find(":selected").val(),
                workflowName: this._workflowModel.get("name")
            };

            request = this._makeRequest(options);
            request.done(this._onSuccess).fail(callback).fail(this._onError);
        },

        /**
         * @returns {object} Data for the view's template.
         */
        serializeData: function () {
            return this._statusModel.toJSON();
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.Dialogs.StatusDialogView", null, require("workflow-designer/dialogs/status-dialog-view"));