AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var CanvasModel = require("workflow-designer/canvas-model");
    var TransitionView = require("workflow-designer/transition-view");
    var StatusView = require("workflow-designer/status-view");
    var WorkflowModel = require("workflow-designer/workflow-model");
    var WorkflowTransitionsAJAXManager = require("workflow-designer/io/ajax/workflow-transitions-ajax-manager");
    var EditTransitionSourceDialogView = require("workflow-designer/dialogs/edit-transition-source-dialog-view");
    var JIRADialog = require("jira/dialog/dialog");
    var TestUtilities = require("workflow-designer/test-utilities");
    var jQuery = require("jquery");

    module("EditTransitionSourceDialogView", {
        setup: function () {
            this.canvas = TestUtilities.testDraw2DCanvas();
            this.sandbox = sinon.sandbox.create();
            this.updateTransitionSourceStub = this.sandbox.stub(WorkflowTransitionsAJAXManager, "updateTransitionSource");

            this.createWorkflowModel();
            this.createStatusModelsAndViews();
            this.createTransitionModelsAndViews();
        },

        createWorkflowModel: function () {
            this.workflowModel = new WorkflowModel({name: "Workflow Name"});
            this.workflowModel.updateTransitionSource = this.sandbox.spy();
        },

        createStatusModelsAndViews: function () {
            this.originalSourceStatus = this.workflowModel.addStatus({stepId: 1, name: "Source"});

            this.sourceView = new StatusView({
                canvas: this.canvas,
                model: this.originalSourceStatus,
                workflowModel: this.workflowModel
            }).render();

            this.targetView = new StatusView({
                canvas: this.canvas,
                model: this.workflowModel.addStatus({stepId: 2, name: "Target"}),
                workflowModel: this.workflowModel
            }).render();

            this.newSourceView = new StatusView({
                canvas: this.canvas,
                model: this.workflowModel.addStatus({stepId: 3, name: "New source"}),
                workflowModel: this.workflowModel
            }).render();

            this.sourcePort = this.newSourceView.getPorts()[0];
        },

        createTransitionModelsAndViews: function () {
            this.transitionModel = this.workflowModel.addTransition({
                actionId: 1,
                source: this.sourceView.model,
                target: this.targetView.model,
                targetAngle: 90
            });

            this.transitionView = new TransitionView({
                canvas: this.canvas,
                canvasModel: new CanvasModel({}, {
                    workflowModel: this.workflowModel
                }),
                model: this.transitionModel,
                sourceView: this.sourceView,
                targetView: this.targetView,
                workflowModel: this.workflowModel
            }).render();

            this.transitionView.resetConnection = this.sandbox.spy();
        },

        /**
         * Create and show a <tt>EditTransitionSourceDialogView</tt>.
         *
         * @return {object} The dialog's element and deferred result.
         */
        showDialog: function () {
            new EditTransitionSourceDialogView({
                newSourcePort: this.sourcePort,
                newSourceView: this.newSourceView,
                originalSourceStatus: this.originalSourceStatus,
                transitionView: this.transitionView,
                workflowModel: this.workflowModel
            }).show();

            return JIRADialog.current && JIRADialog.current.get$popup();
        },

        teardown: function () {
            TestUtilities.removeDialogs();
            this.sandbox.restore();
        }
    });

    test("Editing transition source", function () {
        var deferred = jQuery.Deferred(),
            element = this.showDialog(),
            sourceAngle = this.newSourceView.getAngleToPort(this.sourcePort);

        ok(TestUtilities.dialogIsVisible(), "The dialog is visible");
        ok(element.find(":submit").length, "The dialog has a submit button");

        this.updateTransitionSourceStub.returns(deferred.promise());

        TestUtilities.submit(element);

        equal(element.find(":input:disabled").length, element.find(":input").length, "All inputs are disabled while submitting");
        equal(this.updateTransitionSourceStub.callCount, 1, "WorkflowTransitionsAJAXManager.updateTransitionSource() was called");

        deferred.resolve();

        equal(this.workflowModel.updateTransitionSource.callCount, 1, "WorkflowModel.updateTransitionSource was called");
        deepEqual(this.workflowModel.updateTransitionSource.args[0][0], {
            transition: this.transitionModel,
            sourceAngle: sourceAngle,
            sourceStepId: 3
        }, "It was passed the correct arguments");

        ok(!TestUtilities.dialogIsVisible(), "The dialog closed");
    });

    test("An error message is shown if updating the transition source fails", function () {
        var element = this.showDialog(),
            errorElement,
            errorMessage = "Oh noes!";

        this.updateTransitionSourceStub.returns(jQuery.Deferred().reject(errorMessage).promise());
        TestUtilities.submit(element);

        errorElement = element.find(".aui-message.error");
        equal(errorElement.length, 1, "An error message is shown in the dialog");
        ok(errorElement.text().indexOf(errorMessage) > -1, "It contains the correct text");
        equal(element.find(":input:disabled").length, 0, "No elements in the dialog are disabled");

        this.updateTransitionSourceStub.returns(jQuery.Deferred().promise());
        TestUtilities.submit(element);
        equal(element.find(".error").length, 0, "Errors are removed on submit");
    });

    test("Closing the dialog resets transition target", function () {
        var element = this.showDialog();

        ok(TestUtilities.dialogIsVisible(), "The dialog is visible");

        element.find("#cancel").click();

        equal(this.transitionView.resetConnection.callCount, 1, "Transition target was reset");
        ok(!TestUtilities.dialogIsVisible(), "The dialog closed");
    });

});
