AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var jQuery = require("jquery");
    var StatusesAJAXManager = require("workflow-designer/io/ajax/statuses-ajax-manager");
    var StatusCategoriesAJAXManager = require("workflow-designer/io/ajax/status-categories-ajax-manager");
    var WorkflowStatusesAJAXManager = require("workflow-designer/io/ajax/workflow-statuses-ajax-manager");
    var StatusModel = require("workflow-designer/status-model");
    var WorkflowModel = require("workflow-designer/workflow-model");
    var EditStatusDialogView = require("workflow-designer/dialogs/edit-status-dialog-view");
    var Messages = require("workflow-designer/messages");
    var Analytics = require("workflow-designer/analytics");
    var JIRADialog = require("jira/dialog/dialog");
    var TestUtilities = require("workflow-designer/test-utilities");
    var _ = require("workflow-designer/underscore");

    module("EditStatusDialogView", {
        setup: function () {
            var getStatusCategoriesDeferred,
                getStatusesDeferred,
                sandbox = this.sandbox = sinon.sandbox.create();

            getStatusCategoriesDeferred = jQuery.Deferred().resolve([{
                colorName: "blue-gray",
                id: 2,
                key: "new",
                name: "New"
            }, {
                colorName: "green",
                id: 3,
                key: "done",
                name: "Complete"
            }]);

            getStatusesDeferred = jQuery.Deferred().resolve([{
                id: 100,
                numberOfWorkflows: 1
            }]);

            this.getAssociatedWorkflowsStub = sandbox.stub(StatusesAJAXManager, "getNumberOfAssociatedWorkflows").returns(5);
            this.getStatusesStub = sandbox.stub(StatusesAJAXManager, "getStatuses").returns(getStatusesDeferred);
            this.getStatusCategoriesStub = sandbox.stub(StatusCategoriesAJAXManager, "getStatusCategories").returns(getStatusCategoriesDeferred);
            this.updateStatusStub = sandbox.stub(WorkflowStatusesAJAXManager, "updateStatus").returns(jQuery.Deferred());
            this.workflowModel = new WorkflowModel({
                name: "Workflow Name"
            });
        },

        /**
         * Create and show an `EditStatusDialogView`.
         *
         * @param {object} [options] Options to pass to the dialog's constructor.
         * @return {object} The dialog's element.
         */
        showDialog: function (options) {
            options = _.defaults({}, options, {
                statusModel: new StatusModel({
                    statusCategory: {
                        id: 3
                    },
                    statusId: 100,
                    stepId: 1,
                    name: "Status",
                    description: "This is the description."
                }),
                workflowModel: this.workflowModel
            });

            new EditStatusDialogView(options).show();
            return JIRADialog.current && JIRADialog.current.get$popup();
        },

        teardown: function () {
            TestUtilities.removeDialogs();
            this.sandbox.restore();
        }
    });

    test("A warning is shown if the status is used in other workflows", function () {
        this.getStatusesStub.returns(jQuery.Deferred().resolve([{
            id: 100
        }]));

        equal(this.showDialog().find(".aui-message.warning").length, 1, "A warning is shown");
    });

    test("An error message is shown if editing the status fails", function () {
        var element = this.showDialog(),
            errorElement,
            errorMessage = "Oh noes!";

        this.updateStatusStub.returns(jQuery.Deferred().reject(errorMessage).promise());
        TestUtilities.submit(element);

        errorElement = element.find(".aui-message.error");
        equal(errorElement.length, 1, "An error message is shown in the dialog");
        ok(errorElement.text().indexOf(errorMessage) > -1, "It contains the correct text");
        equal(element.find(":input:disabled").length, 0, "No elements in the dialog are disabled");

        this.updateStatusStub.returns(jQuery.Deferred().promise());
        TestUtilities.submit(element);
        equal(element.find(".error").length, 0, "Errors are removed on submit");
    });

    test("An error message is shown if loading associated workflows fails", function () {
        var errorMessage = "No associated workflows for you!",
            showErrorMessageStub = this.sandbox.stub(Messages, "showErrorMessage");

        this.getAssociatedWorkflowsStub.returns(jQuery.Deferred().reject(errorMessage));

        this.showDialog();
        ok(!TestUtilities.dialogIsVisible(), "The dialog was hidden");
        ok(showErrorMessageStub.calledWithExactly(errorMessage), "An error message was shown");
    });

    test("An error message is shown if loading statuses fails", function () {
        var errorMessage = "No statuses for you!",
            showErrorMessageStub = this.sandbox.stub(Messages, "showErrorMessage");

        this.getStatusesStub.returns(jQuery.Deferred().reject(errorMessage));

        this.showDialog();
        ok(!TestUtilities.dialogIsVisible(), "The dialog was hidden");
        ok(showErrorMessageStub.calledWithExactly(errorMessage), "An error message was shown");
    });

    test("Editing a status", function () {
        var deferred = jQuery.Deferred(),
            element,
            expectedAttributes,
            expectedArguments,
            statusModel,
            statusModelSpy = this.sandbox.spy(StatusModel.prototype, "initialize");

        element = this.showDialog();
        statusModel = statusModelSpy.thisValues[0];

        this.updateStatusStub.returns(deferred.promise());
        this.sandbox.stub(Analytics, "triggerUpdateStatus");
        this.sandbox.stub(Analytics, "triggerUpdateStep");

        TestUtilities.submit(element, {
            "#status-description": "Awesome new description!",
            "#status-name": "New Name"
        });

        equal(element.find(":input:disabled").length, element.find(":input").length, "All inputs are disabled while submitting");
        equal(this.updateStatusStub.callCount, 1, "WorkflowStatusesAJAXManager.updateStatus() was called");

        deferred.resolve();
        expectedArguments = [{
            description: "Awesome new description!",
            name: "New Name",
            statusCategoryId: "3",
            statusId: 100,
            workflowName: "Workflow Name"
        }];
        expectedAttributes = {
            description: "Awesome new description!",
            name: "New Name",
            statusCategory: {
                colourName: "green",
                id: 3
            }
        };

        deepEqual(this.updateStatusStub.args[0], expectedArguments, "It was passed the correct arguments");
        deepEqual(_.pick(statusModel.attributes, _.keys(expectedAttributes)), expectedAttributes,
            "The StatusModel's attributes were updated");

        ok(Analytics.triggerUpdateStatus.calledOnce, "Analytics event for update/status has been triggered");
        sinon.assert.calledWith(Analytics.triggerUpdateStatus, this.workflowModel.get('permissions'));
        ok(Analytics.triggerUpdateStep.calledOnce, "Analytics event for udpate/step has been triggered");
        sinon.assert.calledWith(Analytics.triggerUpdateStep, this.workflowModel.get('permissions'));
    });

    test("Fields are populated with the status's attributes", function () {
        var element = this.showDialog();

        equal(element.find("#status-description").val(), "This is the description.", "The status's description was used");
        equal(element.find("#status-name").val(), "Status", "The status's name was used");
    });

    test("No warning is shown if the status isn't used in other workflows", function () {
        this.getAssociatedWorkflowsStub.returns(jQuery.Deferred().resolve(0));

        equal(this.showDialog().find(".aui-message.warning").length, 0, "No warning is visible");
    });

    test("The selected status category is used", function () {
        var statusModel,
            statusModelSpy = this.sandbox.spy(StatusModel.prototype, "initialize");

        this.updateStatusStub.returns(jQuery.Deferred().resolve());
        TestUtilities.submit(this.showDialog(), {
            "#status-category-id": "2"
        });

        equal(this.updateStatusStub.callCount, 1, "WorkflowStatusesAJAXManager#updateStatus was called");
        equal(this.updateStatusStub.args[0][0].statusCategoryId, "2", "It was passed the correct status category");

        statusModel = statusModelSpy.thisValues[0];
        deepEqual(statusModel.get("statusCategory"), {colourName: "blue-gray", id: 2},
            "The StatusModel's statusCategory attribute was updated");
    });

});
