define("workflow-designer/dialog", [
    "workflow-designer/dialog-size-calculator",
    "jira/dialog/form-dialog",
    "workflow-designer/application-loader",
    "workflow-designer/templates",
    "jira/dialog/dialog",
    "workflow-designer/underscore",
    "jquery"
], function(
    DialogSizeCalculator,
    FormDialog,
    ApplicationLoader,
    Templates,
    JIRADialog,
    _,
    jQuery
) {
    return FormDialog.extend(
    /** @lends JIRA.WorkflowDesigner.Dialog# */
    {
        /**
         * Initialises the dialog.
         *
         * Passes options through to <tt>JIRA.FormDialog</tt>.
         *
         * @constructs
         * @extends JIRA.FormDialog
         * @classdesc JIRA.FormDialog which displays the workflow designer.
         * @param {object} options
         * @param {function|string} [options.currentStepId] The ID of the current step.
         * @param {function|boolean} [options.isDraft=false] Whether the workflow is a draft.
         * @param {function|object} [options.layoutData] The workflow designer's layout data.
         * @param {function|string} [options.workflowId] The ID of the workflow to load.
         */
        init: function (options) {
            _.bindAll(this, "_sizeAndPositionDialog");

            options = _.defaults({}, options, {
                content: this.createDialogContent,
                isDraft: false,
                onContentRefresh: this._showWorkflowDesigner,
                widthClass: "large"
            });

            this.options = options;
            this.sizeAndPositionDialog = _.throttle(this._sizeAndPositionDialog, 100);

            this._super(options);
            this.onContentReady(this._addLabelHint);
        },

        /**
         * Creates the content structure of this dialog.
         *
         * @param {function} callback Callback that we must pass the dialog content to.
         */
        createDialogContent: function(callback) {
            callback(Templates.Dialog.workflowDialog({
                title: _.result(this.options, "workflowId")
            }));
        },

        /**
         * @method
         */
        show: function() {
            this._super.apply(this, arguments);

            jQuery(window).on("resize", this.sizeAndPositionDialog);
        },

        /**
         * @method
         */
        hide: function() {
            this._designer && this._designer.destroy();

            this._super.apply(this, arguments);

            jQuery(window).off("resize", this.sizeAndPositionDialog);
        },

        /**
         * Set the width and height of the current dialog and position it accordingly.
         * Note, this removes the max-height attribute as it causes the dialog to 'jump' when resizing and is
         * not necessary as the height and is managed.
         *
         * @private
         */
        _sizeAndPositionDialog: function () {
            var formBody,
                size;

            if (JIRADialog.current === this) {
                size = DialogSizeCalculator.calculateDialogSize(this);
                this.options.width = size.width;

                formBody = this.$popup.find(".form-body");
                formBody.css({
                    height: size.height,
                    maxHeight: ""
                });

                this._positionInCenter();
            }
        },

        /**
         * Creates the workflow designer.
         *
         * @private
         */
        _showWorkflowDesigner: function() {
            var dialog = this;

            this._sizeAndPositionDialog();

            ApplicationLoader.load(function (Application) {
                if (dialog.isCurrent()) {
                    dialog._designer = new Application({
                        actions: false,
                        element: dialog.get$popup().find(".form-body"),
                        immutable: true,
                        draft: _.result(dialog.options, "isDraft"),
                        layoutData: _.result(dialog.options, "layoutData"),
                        workflowId: _.result(dialog.options, "workflowId"),
                        currentStepId: _.result(dialog.options, "currentStepId")
                    });
                }
            });
        },

        /**
         * Adds a transition label related hint message to the dialog's footer.
         */
        _addLabelHint: function() {
            this.getButtonsContainer().prepend(Templates.Dialog.labelMouseOverHint());
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.Dialog", null, require("workflow-designer/dialog"));