AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var TestUtilities = require("workflow-designer/test-utilities");
    var Analytics = require("workflow-designer/analytics");
    var DeleteTransitionDialogView = require("workflow-designer/dialogs/delete-transition-dialog-view");
    var WorkflowModel = require("workflow-designer/workflow-model");
    var WorkflowGlobalTransitionsAJAXManager = require("workflow-designer/io/ajax/workflow-global-transitions-ajax-manager");
    var WorkflowTransitionsAJAXManager = require("workflow-designer/io/ajax/workflow-transitions-ajax-manager");
    var JIRADialog = require("jira/dialog/dialog");

    module("DeleteTransitionDialogView", {
        setup: function () {
            this.sandbox = sinon.sandbox.create();

            this.deleteTransitionStub = this.sandbox.stub(WorkflowTransitionsAJAXManager, "deleteTransition");
            this.deleteGlobalTransitionStub = this.sandbox.stub(WorkflowGlobalTransitionsAJAXManager, "deleteGlobalTransition");
            this.workflowModel = new WorkflowModel({name: "Workflow Name"});

            this.sourceStatus = this.workflowModel.addStatus({stepId: 1});

            this.transitionModel = this.workflowModel.addTransition({
                actionId: 1,
                source: this.sourceStatus,
                target: this.workflowModel.addStatus({stepId: 2})
            });

            this.globalTransitionModel = this.workflowModel.addTransition({
                actionId: 1,
                globalTransition: true,
                source: this.sourceStatus,
                target: this.workflowModel.addStatus({stepId: 2})
            });

            this.transitionModel.destroy = this.sandbox.spy();
            this.globalTransitionModel.destroy = this.sandbox.spy();
        },

        /**
         * Create and show a <tt>DeleteTransitionDialogView</tt>.
         *
         * @return {object} The dialog's element and deferred result.
         */
        showDialog: function () {
            new DeleteTransitionDialogView({
                workflowModel: this.workflowModel,
                transitionModel: this.transitionModel
            }).show();

            return JIRADialog.current && JIRADialog.current.get$popup();
        },

        /**
         * Create and show a <tt>DeleteTransitionDialogView</tt> for a global transition.
         *
         * @return {object} The dialog's element and deferred result.
         */
        showGlobalTransitionDeleteDialog: function () {
            new DeleteTransitionDialogView({
                workflowModel: this.workflowModel,
                transitionModel: this.globalTransitionModel
            }).show();

            return JIRADialog.current && JIRADialog.current.get$popup();
        },

        teardown: function () {
            TestUtilities.removeDialogs();
            this.sandbox.restore();
        }
    });

    test("Deleting a transition destroys transition model", function () {
        var deferred = jQuery.Deferred(),
            element = this.showDialog();

        ok(TestUtilities.dialogIsVisible(), "The dialog is visible");
        ok(element.find(":submit").length, "The dialog has a submit button");

        this.deleteTransitionStub.returns(deferred.promise());
        this.sandbox.stub(Analytics, "triggerRemoveTransition");

        TestUtilities.submit(element);

        equal(element.find(":input:disabled").length, element.find(":input").length, "All inputs are disabled while submitting");
        equal(this.deleteTransitionStub.callCount, 1, "WorkflowTransitionsAJAXManager.deleteTransition() was called");

        deferred.resolve();

        equal(this.transitionModel.destroy.callCount, 1, "Transition model destroyed");
        ok(!TestUtilities.dialogIsVisible(), "The dialog closed");

        ok(Analytics.triggerRemoveTransition.calledOnce, "Analytics event for remove/transition has been triggered");
        sinon.assert.calledWith(Analytics.triggerRemoveTransition, this.workflowModel.get('permissions'));
    });

    test("Deleting a global transition destroys transition model", function () {
        var deferred = jQuery.Deferred(),
            element = this.showGlobalTransitionDeleteDialog();

        ok(TestUtilities.dialogIsVisible(), "The dialog is visible");
        ok(element.find(":submit").length, "The dialog has a submit button");

        this.deleteGlobalTransitionStub.returns(deferred.promise());
        this.sandbox.stub(Analytics, "triggerRemoveGlobalTransition");

        TestUtilities.submit(element);

        var expectedOptions = {
            "transitionId": 1,
            "sourceStepId": 1,
            "workflowName": "Workflow Name"
        };

        equal(element.find(":input:disabled").length, element.find(":input").length, "All inputs are disabled while submitting");
        equal(this.deleteGlobalTransitionStub.callCount, 1, "WorkflowGlobalTransitionsAJAXManager.deleteGlobalTransition() was called");
        deepEqual(this.deleteGlobalTransitionStub.args[0], [expectedOptions], "It was passed the correct options");

        deferred.resolve();

        equal(this.globalTransitionModel.destroy.callCount, 1, "Transition model destroyed");
        ok(!TestUtilities.dialogIsVisible(), "The dialog closed");

        ok(Analytics.triggerRemoveGlobalTransition.calledOnce, "Analytics event for remove/globaltransition has been triggered");
        sinon.assert.calledWith(Analytics.triggerRemoveGlobalTransition, this.workflowModel.get('permissions'));
    });

    test("An error message is shown if deleting the transition fails", function () {
        var element = this.showDialog(),
            errorElement,
            errorMessage = "Oh noes!";

        this.deleteTransitionStub.returns(jQuery.Deferred().reject(errorMessage).promise());
        TestUtilities.submit(element);

        errorElement = element.find(".aui-message.error");
        equal(errorElement.length, 1, "An error message is shown in the dialog");
        ok(errorElement.text().indexOf(errorMessage) > -1, "It contains the correct text");
        equal(element.find(":input:disabled").length, 0, "No elements in the dialog are disabled");

        this.deleteTransitionStub.returns(jQuery.Deferred().promise());
        TestUtilities.submit(element);
        equal(element.find(".error").length, 0, "Errors are removed on submit");
    });

    test("Closing the dialog doesn't destroy transition model", function () {
        var element = this.showDialog();

        ok(TestUtilities.dialogIsVisible(), "The dialog is visible");

        element.find(".cancel").click();

        equal(this.transitionModel.destroy.callCount, 0, "Transition model was not destroyed");
        ok(!TestUtilities.dialogIsVisible(), "The dialog closed");
    });

    test("Can't delete initial transition", function () {
        this.sourceStatus.set("initial", true);

        var element = this.showDialog();

        ok(TestUtilities.dialogIsVisible(), "The dialog is visible");

        ok(!element.find(":submit").length, "The dialog does not have a submit button");
    });

});
