AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var TestUtilities = require("workflow-designer/test-utilities");
    var WorkflowStatusesAJAXManager = require("workflow-designer/io/ajax/workflow-statuses-ajax-manager");
    var WorkflowModel = require("workflow-designer/workflow-model");
    var StatusModel = require("workflow-designer/status-model");
    var DeleteStatusDialogView = require("workflow-designer/dialogs/delete-status-dialog-view");
    var Analytics = require("workflow-designer/analytics");
    var JIRADialog = require("jira/dialog/dialog");
    var jQuery = require("jquery");

    module("DeleteStatusDialogView", {
        setup: function () {
            this.sandbox = sinon.sandbox.create();

            this.removeStatusStub = this.sandbox.stub(WorkflowStatusesAJAXManager, "removeStatus");
            this.workflowModel = new WorkflowModel({name: "Workflow Name", liveStepIds: [1]});

            this.statusModel = this.workflowModel.addStatus({stepId: 1, statusId: 100});
            this.transitionModel = this.workflowModel.addTransition({
                source: new StatusModel({stepId: 2, statusId: 101, initial: true}),
                target: this.statusModel
            });

            this.statusModel.destroy = this.sandbox.spy();
            this.workflowModel.destroy = this.sandbox.spy();
        },

        /**
         * Create and show an <tt>DeleteStatusDialogView</tt>.
         *
         * @return {object} The dialog's element and deferred result.
         */
        showDialog: function () {
            new DeleteStatusDialogView({
                errorMessage: this.errorMessage,
                workflowModel: this.workflowModel,
                statusModel: this.statusModel
            }).show();

            return JIRADialog.current && JIRADialog.current.get$popup();
        },

        getDialog: function () {
            return JIRADialog.current && JIRADialog.current.get$popup();
        },

        teardown: function () {
            TestUtilities.removeDialogs();
            this.sandbox.restore();
        }
    });

    test("Deleting a status reloads the new workflow layout", function () {
        var element = this.showDialog(),
            expectedArguments,
            layoutData = {},
            workflowModelResetStub = this.sandbox.stub(this.workflowModel, "reset");

        ok(TestUtilities.dialogIsVisible(), "The dialog is visible");

        this.removeStatusStub.returns(jQuery.Deferred().resolve(layoutData).promise());
        this.sandbox.stub(Analytics, "triggerRemoveStep");

        TestUtilities.submit(element);

        expectedArguments = [{
            statusId: 100,
            workflowName: "Workflow Name"
        }];

        equal(this.removeStatusStub.callCount, 1, "WorkflowStatusesAJAXManager.removeStatus() was called");
        deepEqual(this.removeStatusStub.args[0], expectedArguments, "It was passed the correct arguments");

        ok(!TestUtilities.dialogIsVisible(), "No dialog is visible");
        equal(workflowModelResetStub.callCount, 1, "WorkflowModel.reset() was called");
        deepEqual(workflowModelResetStub.args[0], [layoutData], "It was passed the layout data returned from removeStatus()");

        ok(Analytics.triggerRemoveStep.calledOnce, "Analytics event for remove/step has been triggered");
        sinon.assert.calledWith(Analytics.triggerRemoveStep, this.workflowModel.get('permissions'));
    });

    test("An error message is shown if deleting the status fails", function () {
        var element = this.showDialog(),
            errorElement,
            errorMessage = "Oh noes!";

        this.removeStatusStub.returns(jQuery.Deferred().reject(errorMessage).promise());
        TestUtilities.submit(element);

        element = this.getDialog();
        errorElement = element.find(".form-body");
        equal(errorElement.length, 1, "An error message is shown in the dialog");
        ok(errorElement.text().indexOf(errorMessage) > -1, "It contains the correct text");
        equal(element.find(":input").length, 0, "There are no input elements");
    });

    test("An error message is shown if error is passed in options", function () {
        this.errorMessage = "This is test error message";
        var element = this.showDialog();

        var errorElement = element.find(".form-body");
        equal(errorElement.length, 1, "An error message is shown in the dialog");
        ok(errorElement.text().indexOf(this.errorMessage) > -1, "It contains the correct text");
        equal(element.find(":input").length, 0, "There are no input elements");
    });

    test("Closing the dialog doesn't destroy workflow model", function () {
        var element = this.showDialog();

        ok(TestUtilities.dialogIsVisible(), "The dialog is visible");

        element.find(".cancel").click();

        equal(this.workflowModel.destroy.callCount, 0, "Workflow model was not destroyed");
        equal(this.statusModel.destroy.callCount, 0, "Status model was not destroyed");
        ok(!TestUtilities.dialogIsVisible(), "The dialog closed");
    });

    test("Can't delete initial status", function () {
        this.statusModel.set("initial", true);

        var element = this.showDialog();

        ok(TestUtilities.dialogIsVisible(), "The dialog is visible");

        ok(!element.find(":submit").length, "The dialog does not have a submit button");
    });

    test("Can't delete status connected to the initial transition", function () {
        this.transitionModel.get("source").set("initial", true);

        var element = this.showDialog();

        ok(TestUtilities.dialogIsVisible(), "The dialog is visible");

        ok(!element.find(":submit").length, "The dialog does not have a submit button");
    });
});
