define("workflow-designer/zoom-handler", [
    "workflow-designer/backbone",
    "workflow-designer/underscore",
    "jquery"
], function(
    Backbone,
    _,
    jQuery
) {

    /**
     * @classdesc Observes and reports zoom gestures.
     * @constructs JIRA.WorkflowDesigner.ZoomHandler
     * @extends Backbone.Events
     * @param {element} canvas The workflow designer's canvas element.
     */
    var ZoomHandler = function (canvas) {
        _.bindAll(this, "_onMouseWheel");

        jQuery(document).on({
            mousewheel: this._onMouseWheel,
            MozMousePixelScroll: this._onMouseWheel // Firefox
        });

        this._canvas = jQuery(canvas).get(0);
    };

    _.extend(ZoomHandler.prototype, Backbone.Events,
    /** @lends JIRA.WorkflowDesigner.ZoomHandler.prototype*/
    {
        /**
         * Compute the zoom factor of a mouse wheel movement.
         *
         * @param {object} e The mouse wheel event that triggered this.
         * @private
         * @returns {number} The computed zoom factor.
         */
        _computeZoomFactor: function (e) {
            var isFirefox,
                minimumZoomFactor = 0.1,
                zoomFactor;

            e = e.originalEvent;
            isFirefox = !!e.detail;

            if (isFirefox) {
                zoomFactor = this._getDeltaFirefox(e);
            } else {
                zoomFactor = this._getDeltaOthers(e);
            }

            isNaN(zoomFactor) && (zoomFactor = 1);
            return Math.max(minimumZoomFactor, zoomFactor);
        },

        /**
         * Destroy the zoom handler, unbinding its event handlers, etc.
         */
        destroy: function () {
            jQuery(document).off({
                mousewheel: this._onMouseWheel,
                MozMousePixelScroll: this._onMouseWheel
            });
        },

        /**
         * Get the zoom delta from a scroll event triggered in Firefox.
         *
         * @param {Event} e The scroll event that was triggered.
         * @private
         * @returns {number} The zoom delta extracted from `e`.
         */
        _getDeltaFirefox: function (e) {
            var dampening = 200,
                delta = e.axis === e.VERTICAL_AXIS && e.detail || 0;

            return 1 + delta / dampening;
        },

        /**
         * Get the zoom delta from a scroll event triggered in a browser other than Firefox.
         *
         * @param {Event} e The scroll event that was triggered.
         * @private
         * @returns {number} The zoom delta extracted from `e`.
         */
        _getDeltaOthers: function (e) {
            var dampening = 500,
                delta = e.wheelDelta;

            if ("wheelDeltaY" in e) {
                delta = e.wheelDeltaY;
            }

            return 1 - delta / dampening;
        },

        /**
         * @param {element} element An element.
         * @private
         * @returns {boolean} Whether `element` is inside the workflow designer.
         */
        _isWorkflowDesigner: function (element) {
            return element === this._canvas || jQuery.contains(this._canvas, element);
        },

        /**
         * Respond to a mouse wheel event.
         *
         * @param {jQuery.Event} e The mouse wheel event.
         * @private
         */
        _onMouseWheel: function (e) {
            var isZoom = e.altKey && this._isWorkflowDesigner(e.target);
            if (isZoom) {
                e.preventDefault();

                this.trigger("zoom", {
                    clientX: e.originalEvent.clientX,
                    clientY: e.originalEvent.clientY,
                    factor: this._computeZoomFactor(e)
                });
            }
        }
    });

    return ZoomHandler;
});

AJS.namespace("JIRA.WorkflowDesigner.ZoomHandler", null, require("workflow-designer/zoom-handler"));