define("workflow-designer/positioner", [
    "workflow-designer/draw-2d",
    "workflow-designer/underscore"
], function(
    draw2d,
    _
) {
    /**
     * Gap between automatically positioned statuses.
     *
     * @inner
     * @constant
     * @type {number}
     */
    var gapBetweenStatuses = 20;

    /**
     * Calculates and sets status positions.
     *
     * @namespace
     */
    var Positioner = {

        positionLoopedTransitionContainer: function (options) {
            var loopedTransitionContainerFigure,
                maxY,
                minX,
                minY,
                newX,
                newY;

            if (options.loopedTransitionContainerView.model.hasCoordinates()) {
                return;
            }

            options.statusViews.each(function(statusView) {
                var statusBoundingBox = statusView.getBoundingBox(),
                    statusBottom = statusBoundingBox.getBottom(),
                    statusLeft = statusBoundingBox.getLeft(),
                    statusTop = statusBoundingBox.getTop();

                if (!minX || minX > statusLeft) {
                    minX = statusLeft;
                }
                if (!minY || minY > statusTop) {
                    minY = statusTop;
                }
                if (!maxY || maxY < statusBottom) {
                    maxY = statusBottom;
                }
            });

            loopedTransitionContainerFigure = options.loopedTransitionContainerView.getFigure();
            newX = minX - 100 - loopedTransitionContainerFigure.getWidth();
            newY = minY + (maxY - minY)/2 - loopedTransitionContainerFigure.getHeight()/2;

            options.loopedTransitionContainerView.setPosition(newX, newY);
            options.loopedTransitionContainerView.resetFigure();
        },

        /**
         * Position statuses that don't have coordinates yet.
         *
         * @param {object} options
         * @param {JIRA.WorkflowDesigner.StatusView[]} options.statusViews The statuses to position.
         * @param {draw2d.geo.Rectangle} options.viewBox The canvas's view box.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The workflow model.
         * @return {JIRA.WorkflowDesigner.StatusView[]} Positioned statuses.
         */
        positionStatuses: function (options) {
            var positionCentre,
                positionedStatuses,
                top;

            positionCentre = new draw2d.geo.Point(
                options.viewBox.getCenter().getX(),
                options.viewBox.getY() + options.viewBox.getHeight() * 0.2
            );

            function calculateTop(statusViews) {
                var groupHeight = _.sum(_.map(statusViews, getHeight));
                top = positionCentre.getY() - groupHeight / 2;
            }

            function positionStatusView(statusView) {
                statusView.setPosition(positionCentre.getX() - statusView.getBoundingBox().getWidth() / 2, top);
                top += getHeight(statusView);
            }

            positionedStatuses = _.chain(options.statusViews).reject(hasCoordinates).value();
            positionedStatuses.sort(statusViewComparator(options.workflowModel));
            _.chain(positionedStatuses).tap(calculateTop).each(positionStatusView);
            return positionedStatuses;
        }
    };

    /**
     * @inner
     * @param {JIRA.WorkflowDesigner.WorkflowModel} workflowModel The workflow model.
     * @return {function} A comparator function for sorting automatically positioned statuses.
     */
    function statusViewComparator(workflowModel) {
        var initialTransitionDestinations = workflowModel.getInitialTransitionDestinations();

        return function(statusViewOne, statusViewTwo) {
            var statusOneIsInitial,
                statusOneIsInitialTarget,
                statusOneModel = statusViewOne.model,
                statusTwoIsInitial,
                statusTwoIsInitialTarget,
                statusTwoModel = statusViewTwo.model;

            statusOneIsInitial = statusOneModel.isInitial();
            statusTwoIsInitial = statusTwoModel.isInitial();

            // Initial statuses go before ordinary ones.
            if (statusOneIsInitial && !statusTwoIsInitial) {
                return -1;
            }
            if (!statusOneIsInitial && statusTwoIsInitial) {
                return 1;
            }

            if (!statusOneIsInitial && !statusTwoIsInitial) {
                // Initial transition destination statuses go before other statuses.
                statusOneIsInitialTarget = _.contains(initialTransitionDestinations, statusOneModel);
                statusTwoIsInitialTarget = _.contains(initialTransitionDestinations, statusTwoModel);

                if (statusOneIsInitialTarget && !statusTwoIsInitialTarget) {
                    return -1;
                }
                if (!statusOneIsInitialTarget && statusTwoIsInitialTarget) {
                    return 1;
                }
            }

            // Otherwise just order based on status ID.
            return statusOneModel.get("id").localeCompare(statusTwoModel.get("id"));
        };
    }

    /**
     * @inner
     * @param {JIRA.WorkflowDesigner.StatusView} statusView A <tt>StatusView</tt>.
     * @return {number} <tt>statusView</tt>'s height.
     */
    function getHeight(statusView) {
        return statusView.getBoundingBox().getHeight() + gapBetweenStatuses;
    }

    /**
     * @inner
     * @param {JIRA.WorkflowDesigner.StatusView} statusView A <tt>StatusView</tt>.
     * @return {boolean} Whetheer <tt>statusView</tt>'s model has coordinates.
     */
    function hasCoordinates(statusView) {
        return statusView.model.hasCoordinates();
    }

    return Positioner;
});

AJS.namespace("JIRA.WorkflowDesigner.Positioner", null, require("workflow-designer/positioner"));