AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:global-workflows",
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var GlobalWorkflows = require("workflow-designer/global-workflows");
    var DarkFeatures = require("jira/ajs/dark-features");
    var TestUtilities = require("workflow-designer/test-utilities");
    var TestMocks = require("workflow-designer/test-mocks");
    var jQuery = require("jquery");
    var _ = require("workflow-designer/underscore");

    module("GlobalWorkflows", {
        setup: function () {
            this.sandbox = sinon.sandbox.create();
            this.Application = TestMocks.Application();
            TestMocks.load(this.sandbox, this.Application);

            this.qunitFixture = jQuery("#qunit-fixture");
            this.workflowLinks = jQuery("<div id='workflow-links'>").appendTo(this.qunitFixture);
            this.diagramLink = jQuery("<div id='workflow-diagram'>").appendTo(this.workflowLinks);
            this.textLink = jQuery("<div id='workflow-text'>").appendTo(this.workflowLinks);

            this.isOnDemand = this.sandbox.stub(DarkFeatures, "isEnabled").withArgs("com.atlassian.jira.config.CoreFeatures.ON_DEMAND").returns(false);
        },

        teardown: function () {
            this.sandbox.restore();
        },

        /**
         * @param {object} options
         * @param {boolean} [options.isEditable] Edit mode or view mode.
         * @private
         */
        _setupWorkflowData: function (options) {
            options = _.defaults({}, options, {isDraft: true});

            options.isDraft && this.qunitFixture.append(jQuery("<span class='status-draft'>"));
            options.isInactive && this.qunitFixture.append(jQuery("<span class='status-inactive'>"));
            this.qunitFixture.append(jQuery("<span class='workflow-name'>workflow<span>"));
            this.qunitFixture.append(jQuery("<input id='workflow-designer-project' value='1000'>"));

            if (options.isEditable) {
                this.qunitFixture.append(jQuery("<span id='edit-workflow-trigger'>"));
            }
        },

        initEditMode: function (options) {
            this._createWorkflowView();

            jQuery("<div id='workflow-canvas'>").appendTo(this.workflowView);

            this.init(_.extend({}, options, {
                isEditable: true
            }));
        },

        /**
         * @param {object} options
         * @param {boolean} [options.diagramModeActive] Is diagram mode selected initially.
         */
        initViewMode: function (options) {
            options = options || {};

            this._createWorkflowView();

            this.diagramLink.addClass("workflow-view-toggle");
            this.textLink.addClass("workflow-view-toggle");

            if (options.diagramModeActive) {
                this.diagramLink.addClass("active");
            } else {
                this.workflowView.addClass("hidden");
            }

            this.init(options);
        },

        init: function (options) {
            this._setupWorkflowData(options);

            new GlobalWorkflows();
        },

        _createWorkflowView: function () {
            this.workflowView = jQuery("<div id='workflow-view-diagram'>").appendTo(this.qunitFixture);
        },

        testAnalyticsEvent: function (testCallback) {
            var existingEventQueue = AJS.EventQueue;

            try {
                AJS.EventQueue = {
                    push: sinon.spy()
                };

                testCallback.call(this, AJS.EventQueue.push);
            } finally {
                AJS.EventQueue = existingEventQueue;
            }
        }
    });

    test("Workflow designer is not created in text edit mode", function () {
        this.init({isEditable: true});

        equal(this.Application.callCount, 0, "Workflow designer not initialised");
    });

    test("Workflow designer is created in edit page", function () {
        var expectedArguments;

        this.initEditMode();

        expectedArguments = {
            draft: true,
            immutable: false,
            workflowId: "workflow"
        };

        equal(this.Application.callCount, 1, "Workflow designer initialised");
        deepEqual(_.pick(this.Application.args[0][0], _.keys(expectedArguments)), expectedArguments, "It was passed the correct arguments");
    });


    test("Workflow designer is created in view page if diagram is selected initially", function () {
        var canvas, expectedArguments;

        this.initViewMode({diagramModeActive: true});

        canvas = this.workflowView.find("#workflow-canvas");
        equal(canvas.length, 1, "Canvas element was added");
        ok(canvas.is(":visible"), "Canvas element is visible");

        expectedArguments = {
            draft: true,
            immutable: true,
            workflowId: "workflow"
        };

        equal(this.Application.callCount, 1, "Workflow designer initialised");
        deepEqual(_.pick(this.Application.args[0][0], _.keys(expectedArguments)), expectedArguments, "It was passed the correct arguments");
    });

    test("Workflow designer is created on view page when diagram is first selected (if not selected initially)", function () {
        var canvas, expectedArguments;

        this.initViewMode();

        canvas = this.workflowView.find("#workflow-canvas");
        equal(canvas.length, 1, "Canvas element was added");
        ok(canvas.is(":hidden"), "Canvas element is hidden");

        equal(this.Application.callCount, 0, "Workflow designer not initialised initially");

        this.diagramLink.click();

        expectedArguments = {
            draft: true,
            immutable: true,
            workflowId: "workflow"
        };

        equal(this.Application.callCount, 1, "Workflow designer initialised");
        deepEqual(_.pick(this.Application.args[0][0], _.keys(expectedArguments)), expectedArguments, "It was passed the correct arguments");
    });


    test("Feedback button is NOT added in edit mode in OnDemand", function () {
        this.isOnDemand.returns(true);

        this.initEditMode();

        equal(this.workflowLinks.find("#workflow-designer-feedback-trigger").length, 0, "Feedback button was NOT added");
    });

    test("Feedback button is NOT added in view mode in OnDemand", function () {
        this.isOnDemand.returns(true);

        this.initViewMode();

        equal(this.workflowLinks.find("#workflow-designer-feedback-trigger").length, 0, "Feedback button was NOT added");
    });

    test("Feedback button is not added in edit mode BTF", function () {
        this.initEditMode();

        equal(this.workflowLinks.find("#workflow-designer-feedback-trigger").length, 0, "Feedback button was not added");
    });

    test("Feedback button is not added in view mode BTF", function () {
        this.initViewMode();

        equal(this.workflowLinks.find("#workflow-designer-feedback-trigger").length, 0, "Feedback button was not added");
    });

    test("Feedback button is not added in IE9", function () {
        var instance = this;

        this.isOnDemand.returns(true);

        TestUtilities.asIE(9, function () {
            instance.initEditMode();

            equal(instance.workflowLinks.find("#workflow-designer-feedback-trigger").length, 0, "Feedback button was not added");
        });
    });

    test("Message is shown instead of the workflow designer in edit mode in IE9", function () {
        var instance = this;

        TestUtilities.asIE(9, function () {
            instance.initEditMode();

            equal(instance.Application.callCount, 0, "Workflow designer not initialised");
            equal(instance.workflowView.find(".aui-message").length, 1, "Message is shown");
        });
    });

    test("Open view mode analytics events is triggered upon opening if the diagram is selected initially", function () {
        this.testAnalyticsEvent(function (pushEvent) {
            this.initViewMode({diagramModeActive: true});

            equal(pushEvent.callCount, 1, "Triggered a single event");
            equal(pushEvent.args[0][0].name, "administration.workflow.open.view.global", "It was an open view mode event");
        });
    });

    test("Open view mode analytics events is triggered upon switching to diagram if the diagram is not selected initially", function () {
        this.testAnalyticsEvent(function (pushEvent) {
            this.initViewMode();
            this.diagramLink.click();

            equal(pushEvent.callCount, 1, "Triggered a single event");
            equal(pushEvent.args[0][0].name, "administration.workflow.open.view.global", "It was an open view mode event");
        });
    });

    test("Open view mode analytics events is not triggered upon opening inactive workflow", function () {
        this.testAnalyticsEvent(function (pushEvent) {
            this.initViewMode({diagramModeActive: true, isInactive: true});

            equal(pushEvent.callCount, 0, "Event not triggered");
        });
    });

    test("Open edit mode analytics events is triggered upon opening the draft workflow", function () {
        this.testAnalyticsEvent(function (pushEvent) {
            this.initEditMode();

            equal(pushEvent.callCount, 1, "Triggered a single event");
            equal(pushEvent.args[0][0].name, "administration.workflow.open.edit.global", "It was an open edit mode event");
        });
    });

    test("Open edit mode analytics events is not triggered upon opening non-draft workflow", function () {
        this.testAnalyticsEvent(function (pushEvent) {
            this.initEditMode({isDraft: false});

            equal(pushEvent.callCount, 0, "Event not triggered");
        });
    });

});
