AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var ViewBoxTransformer = require("workflow-designer/viewbox-transformer");
    var Vector2D = require("workflow-designer/vector-2d");
    var draw2d = require("workflow-designer/draw-2d");
    var _ = require("workflow-designer/underscore");

    module("ViewBoxTransformer", {
        /**
         * Creates a new draw2d.geo.Rectangle with the specified values.
         *
         * This function is just an alias for `new draw2d.geo.Rectangle()`
         *
         * @param x X-coordinate for the rectangle's origin
         * @param y Y-coordinate for the rectangle's origin
         * @param w Rectangle's width
         * @param h Rectangle's height
         * @returns {draw2d.geo.Rectangle} Created rectangle
         */
        rectangle: function (x, y, w, h) {
            return new draw2d.geo.Rectangle(x, y, w, h);
        },

        /**
         * Generate a new viewBox using a displacement of (dx,dy)
         * @param {number} dx Horizontal displacement
         * @param {number} dy Horizontal displacement
         * @param {draw2d.geo.Rectangle} viewBox Current view box
         * @returns {draw2d.geo.Rectangle} ViewBox with the requested displacement
         */
        pan: function (dx, dy, viewBox) {
            viewBox = viewBox || this.rectangle(-100, -100, 300, 300);

            return ViewBoxTransformer.getPanViewBox({
                diagramBoundingBox: this.rectangle(0, 0, 100, 100),
                displacement: new Vector2D(dx, dy),
                viewBox: viewBox
            });
        },

        /**
         * Helper function to do a zoom in/out on the viewport's center
         *
         * @param {number} factor Factor to apply (>1 make the viewport bigger, <1 make it smaller)
         * @param {number} [currentZoom=1] Current zoom of the viewport, used to compute limits
         * @returns {draw2d.geo.Rectangle} New viewport with the zoom transformation applied
         */
        centerZoom: function (factor, currentZoom) {
            return ViewBoxTransformer.getZoomViewBox({
                diagramBoundingBox: this.rectangle(-50, -50, 100, 100),
                factor: factor,
                target: new draw2d.geo.Point(0, 0),
                viewBox: this.rectangle(-100, -100, 200, 200),
                zoom: currentZoom || 1
            });
        },

        /**
         * Helper function to do a zoom in (factor 2) to a point
         *
         * @param {number} x X-coordinate of the target point
         * @param {number} y Y-coordinate of the target point
         * @returns {draw2d.geo.Rectangle} New viewport with the zoom transformation applied
         */
        zoomTo: function (x, y) {
            return ViewBoxTransformer.getZoomViewBox({
                diagramBoundingBox: this.rectangle(-50, -50, 100, 100),
                factor: 2,
                target: new draw2d.geo.Point(x, y),
                viewBox: this.rectangle(-100, -100, 200, 200),
                zoom: 1
            });
        }
    });

    test("getPanViewBox()", function () {
        var tests = [
            [[0, 0], [-100, -100, 300, 300], "Panning with zero displacement does not modify the viewBox"],
            [[10, -15], [-90, -115, 300, 300], "The viewBox can be displaced"],
            [[150, 150], [50, 50, 300, 300], "The diagram can be partially hidden"],

            [[175, 175], [75, 75, 300, 300], "Displacements that hide 75% of the diagram are valid (diagram's bottom-right corner visible)"],
            [[175, -175], [75, -275, 300, 300], "Displacements that hide 75% of the diagram are valid (diagram's top-right corner visible)"],
            [[-175, 175], [-275, 75, 300, 300], "Displacements that hide 75% of the diagram are valid (diagram's bottom-left corner visible)"],
            [[-175, -175], [-275, -275, 300, 300], "Displacements that hide 75% of the diagram are valid (diagram's top-left corner visible)"],

            [[200, 200], [75, 75, 300, 300], "Displacements that hide more than 75% of the diagram are limited (diagram's bottom-right corner still visible)"],
            [[200, -200], [75, -275, 300, 300], "Displacements that hide more than 75% of the diagram are limited (diagram's top-right corner still visible)"],
            [[-200, 200], [-275, 75, 300, 300], "Displacements that hide more than 75% of the diagram are limited (diagram's bottom-left corner still visible)"],
            [[-200, -200], [-275, -275, 300, 300], "Displacements that hide more than 75% of the diagram are limited (diagram's top-left corner still visible)"]
        ];

        _.each(tests, function (test) {
            var actual = this.pan.apply(this, test[0]).toJSON(),
                expected = this.rectangle.apply(this, test[1]).toJSON();

            deepEqual(actual, expected, test[2]);
        }, this);
    });

    test("Displacements when view box is smaller than the default visible margin are limited to diagram boundaries (plus padding)", function () {
        var tests = [
            [[100, 100, this.rectangle(0, 0, 10, 10)], [100, 100, 10, 10], "Diagram's bottom-right corner still visible"],
            [[100, -100, this.rectangle(0, 0, 10, 10)], [100, -10, 10, 10], "Diagram's top-right corner still visible"],
            [[-100, 100, this.rectangle(0, 0, 10, 10)], [-10, 100, 10, 10], "Diagram's bottom-left corner still visible"],
            [[-100, -100, this.rectangle(0, 0, 10, 10)], [-10, -10, 10, 10], "Diagram's top-left corner still visible"]
        ];

        _.each(tests, function (test) {
            var actual = this.pan.apply(this, test[0]).toJSON(),
                expected = this.rectangle.apply(this, test[1]).toJSON();

            deepEqual(actual, expected, test[2]);
        }, this);
    });

    test("getZoomViewBox() applies the new zoom factor", function () {
        var tests = [
            [[1], [-100, -100, 200, 200], "Zooming to the center with factor 1 does not modify the viewBox"],
            [[2], [-200, -200, 400, 400], "Zooming to the center with factor 2 makes the viewBox twice as big"],
            [[0.5], [-50, -50, 100, 100], "Zooming to the center with factor 0.5 makes the viewBox twice as small"],

            [[0.9999], [-100, -100, 200, 200], "Very small changes has no effect"],

            [[1.1, 0.5], [-100, -100, 200, 200], "If the current zoom level is 0.5, trying to zoom out (i.e. make viewport bigger) does nothing"],
            [[0.9, 2], [-100, -100, 200, 200], "If the current zoom level is 2, trying to zoom in (i.e. make diagram smaller) does nothing"]
        ];

        _.each(tests, function (test) {
            var actual = this.centerZoom.apply(this, test[0]).toJSON(),
                expected = this.rectangle.apply(this, test[1]).toJSON();

            deepEqual(actual, expected, test[2]);
        }, this);

        deepEqual(this.centerZoom(0.1, 0.5).toJSON(), this.centerZoom(0.25, 0.5).toJSON(),
            "If the current zoom level is 0.5 (i.e. viewBox is small), zoom factor is limited to 0.25 (i.e. make the viewBox 4x times bigger");
        deepEqual(this.centerZoom(50, 2).toJSON(), this.centerZoom(4, 2).toJSON(),
            "If the current zoom level is 2 (i.e. viewBox is big), zoom factor is limited to 4 (i.e. make the viewBox 4x times smaller");
    });

    test("getZoomViewBox() ensures a portion of the diagram is always visible", function () {
        var tests = [
            [[20, -20], [-220, -180, 400, 400], "Zooming to a point P scales the viewBox and moves it so the point P is now on the center"],

            [[-275, -275], [25, 25, 400, 400], "Zooms that hide 75% of the diagram are valid (diagram's bottom-right corner visible)"],
            [[-275, 275], [25, -425, 400, 400], "Zooms that hide 75% of the diagram are valid (diagram's top-right corner visible)"],
            [[275, -275], [-425, 25, 400, 400], "Zooms that hide 75% of the diagram are valid (diagram's bottom-left corner visible)"],
            [[275, 275], [-425, -425, 400, 400], "Zooms that hide 75% of the diagram are valid (diagram's top-left corner visible)"],

            [[-300, -300], [25, 25, 400, 400], "Zooms that hide more than 75% of the diagram are limited (diagram's bottom-right corner still visible)"],
            [[-300, 300], [25, -425, 400, 400], "Zooms that hide more than 75% of the diagram are limited (diagram's top-right corner still visible)"],
            [[300, -300], [-425, 25, 400, 400], "Zooms that hide more than 75% of the diagram are limited (diagram's bottom-left corner still visible)"],
            [[300, 300], [-425, -425, 400, 400], "Zooms that hide more than 75% of the diagram are limited (diagram's top-left corner still visible)"]
        ];

        _.each(tests, function (test) {
            var actual = this.zoomTo.apply(this, test[0]).toJSON(),
                expected = this.rectangle.apply(this, test[1]).toJSON();

            deepEqual(actual, expected, test[2]);
        }, this);
    });

});
