define("workflow-designer/dialog-size-calculator", [
    "workflow-designer/underscore",
    "jquery"
], function(
    _,
    jQuery
) {

    /**
     * An object for calculating the size of the workflow designer dialog.
     *
     * @namespace
     */
    return {
       /**
        * The distance from the edge of the window for size calculations.
        *
        * @type {number}
        * @constant
        * @default
        */
        DISTANCE_FROM_EDGE: 100,

        /**
         * Calculate the dialog size to make it fill the available space.
         *
         * @param {object} dialog The dialog to calculate size for.
         * @param {object} options
         * @param {number} [options.windowHeight] The height of the window.
         * @param {number} [options.windowWidth] The width of the window.
         * @param {number} [options.dialogHeaderHeight] The height of the dialog header.
         * @param {number} [options.dialogFooterHeight] The height of the dialog footer.
         * @returns {{height: number, width: number}}
         */
        calculateDialogSize: function(dialog, options) {
            this.dialog = dialog;
            options = _.defaults(options || {}, {
                windowHeight: this._getWindowHeight(),
                windowWidth: this._getWindowWidth(),
                dialogHeaderHeight: this._getDialogHeaderHeight(),
                dialogFooterHeight: this._getDialogFooterHeight()
            });

            return {
                height: this._calculateDialogHeight(options.windowHeight, options.dialogHeaderHeight, options.dialogFooterHeight),
                width: this._calculateDialogWidth(options.windowWidth)
            };
        },

        /**
         * Calculate the dialog height to be 100 pixels from the edge of the window on each side.
         * @returns {number} The height
         * @private
         */
        _calculateDialogHeight: function(windowHeight, dialogHeaderHeight, dialogFooterHeight) {
            return windowHeight - dialogHeaderHeight - dialogFooterHeight - (this.DISTANCE_FROM_EDGE * 2);
        },

        /**
         * Calculate the dialog width to be 100 pixels from the edge of the window on each side.
         * @returns {number} The width
         * @private
         */
        _calculateDialogWidth: function(windowWidth) {
            return windowWidth - (this.DISTANCE_FROM_EDGE * 2);
        },

        /**
         * Get the window height.
         * @returns {number}
         * @private
         */
        _getWindowHeight: function() {
            return jQuery(window).height();
        },

        /**
         * Get the window width.
         * @returns {number}
         * @private
         */
        _getWindowWidth: function() {
            return jQuery(window).width();
        },

        /**
         * Get the dialog header height.
         * @returns {number}
         * @private
         */
        _getDialogHeaderHeight: function() {
            return this.dialog.$popupHeading.outerHeight();
        },

        /**
         * Get the dialog header width.
         * @returns {number}
         * @private
         */
        _getDialogFooterHeight: function() {
            return this.dialog.$buttonContainer.outerHeight();
        }
    };
});

AJS.namespace("JIRA.WorkflowDesigner.DialogSizeCalculator", null, require("workflow-designer/dialog-size-calculator"));