define("workflow-designer/browser-zoom-checker", [
    "workflow-designer/messages",
    "workflow-designer/browser-utilities",
    "workflow-designer/underscore",
    "jquery"
], function(
    Messages,
    BrowserUtilities,
    _,
    jQuery
) {
    /**
     * @constructor
     * @classdesc Monitors browser zoom level and notifies the user when it is not 100%.
     */
    var BrowserZoomChecker = function() {
        _.bindAll(this, "_startWindowResizeSession", "_endWindowResizeSession");

        jQuery(window).on("resize", this._startWindowResizeSession);
        this._debouncedEndWindowResizeSession = _.debounce(this._endWindowResizeSession, 250);

        this._intervalId = setInterval(_.bind(this._check, this), 1000);
    };

    _.extend(BrowserZoomChecker.prototype,
    /** @lends JIRA.WorkflowDesigner.BrowserZoomChecker# */
    {
        /**
         * Whether the user agent is running on Mac
         *
         * @type {boolean}
         * @readonly
         */
        _isMac: BrowserUtilities.isMac(),

        /**
         * Agent specific zoom level error message
         * @type {String}
         * @readonly
         */
        _msgErrorZoomLevel: BrowserUtilities.isMac() ? AJS.I18n.getText("workflow.designer.reset.zoom.level.mac") : AJS.I18n.getText("workflow.designer.reset.zoom.level.pc"),

        /**
         * Displays an error message if the current browser zoom level is not supported.
         * If the level is supported and a message was previously shown then clears the message.
         *
         * @private
         */
        _check: function() {

            // WebKit browsers seem to update window.outerWidth at a slower rate than window.innerWidth while window is being rapidly re-sized.
            // Disable zoom checking until window resizing is finished.
            if (this._windowIsResizing) {
                return;
            }

            var zoomLevel = BrowserUtilities.getZoomLevel();

            if(!zoomLevel || zoomLevel === 1) {
                this._destroyErrorMessage();
            } else {
                this._showErrorMessage(this._msgErrorZoomLevel);
            }
        },

        /**
         * Stops the browser zoom level checking.
         */
        destroy: function() {
            clearInterval(this._intervalId);
            jQuery(window).off("resize", this._startWindowResizeSession);
            this._destroyErrorMessage();
        },

        isErrorMessageDisplayed: function () {
            return this._browserZoomMessage && this._browserZoomMessage.is(":visible");
        },

        _destroyErrorMessage: function() {
            if (this._browserZoomMessage) {
                this._browserZoomMessage.remove();
                delete this._browserZoomMessage;
            }
        },

        _showErrorMessage: function(message) {
            if (!this._browserZoomMessage) {
                this._browserZoomMessage = Messages.showErrorMessage(message);
            }
        },

        /**
         * Handler for end window resize session.
         *
         * @private
         */
        _endWindowResizeSession: function() {
            this._windowIsResizing = false;
            this._destroyErrorMessage();
        },

        /**
         * Handler for start window resize session.
         *
         * @private
         */
        _startWindowResizeSession: function() {
            this._windowIsResizing = true;
            this._debouncedEndWindowResizeSession();
        }
    });

    return BrowserZoomChecker;
});

AJS.namespace("JIRA.WorkflowDesigner.BrowserZoomChecker", null, require("workflow-designer/browser-zoom-checker"));