define("workflow-designer/analytics", [
    "workflow-designer/io/ajax/user-properties-ajax-manager",
    "jquery"
], function(
    UserPropertiesAJAXManager,
    jQuery
) {
    var addedFirstStatus, addedFirstTransition;

    function getPrefix(permissions) {
        if (!permissions) {
            return "";
        }
        if (!permissions.get('administrator') && permissions.get('editWorkflow')) {
            return "editworkflowpermission.";
        }
        return "";
    }

    /**
     * Triggers a workflow analytic event with the passed name and properties.
     *
     * @inner
     * @param {string} name Event name.
     * @param {object} properties Event properties.
     */
    function triggerEvent(name, properties, permissions) {
        if (AJS.EventQueue) {
            var prefix = getPrefix(permissions);
            AJS.EventQueue.push({
                name: "administration.workflow." + prefix + name,
                properties: properties
            });
        }
    }

    /**
     * Triggers a workflow analytic safe event for privacy policy with the passed name and properties.
     *
     * @inner
     * @param {string} name Event name.
     * @param {object} properties Event properties.
     */
    function triggerSafeEvent(name, properties) {
        var eventName = "administration.workflow." + name;

        if (AJS.Analytics) {
            AJS.Analytics.triggerPrivacyPolicySafeEvent(eventName, properties);
        }
        else {
            AJS.trigger('analyticsEvent', {name: eventName, data: properties});
        }
    }

    /**
     * Attaches a click handler that will trigger the specified analytics safe event
     * when an element matching the specified selector is clicked.
     *
     * @inner
     * @param {string} selector jQuery selector of the element to match.
     * @param {string} eventName Name of the event to trigger when the matching element is clicked.
     */
    function attachOnClickSafeEventTrigger(selector, eventName) {
        jQuery(document).on("click", selector, function() {
            triggerSafeEvent(eventName);
        });
    }

    /**
     * Triggers the Analytics event 'administration.workflow.edit' with the properties:
     *
     * * mode: 'new-diagram'
     * * action: <action param>
     * * object: <object param>
     *
     * @inner
     * @param {string} action Value to use in the 'action' property
     * @param {string} object Value to use in the 'object' property
     */
    function triggerEditEvent(action, object, permissions, analyticsCommonData) {
        // This could have been handled by JIRA's analytics.js, using the custom event
        // "wfd-edit-action". However, that method always uses properties.mode="diagram"
        // so we need to add the event to the EventQueue ourselves.
        triggerEvent("edit", jQuery.extend({
            mode: "new-diagram",
            action: action,
            object: object
        },
        analyticsCommonData),
        permissions);
    }

    attachOnClickSafeEventTrigger(".enter-full-screen", "fullscreen");
    attachOnClickSafeEventTrigger("[data-transition-option-key='properties.key']", "view.transition.properties");
    attachOnClickSafeEventTrigger("[data-transition-option-key='com.atlassian.jira.plugins.jira-transition-triggers-plugin:triggers-transitions-tab']", "view.transition.triggers-transitions-tab");
    attachOnClickSafeEventTrigger("[data-transition-option-key='com.atlassian.jira.plugin.system.workfloweditor.transition.tabs:conditions']", "view.transition.conditions");
    attachOnClickSafeEventTrigger("[data-transition-option-key='com.atlassian.jira.plugin.system.workfloweditor.transition.tabs:validators']", "view.transition.validators");
    attachOnClickSafeEventTrigger("[data-transition-option-key='com.atlassian.jira.plugin.system.workfloweditor.transition.tabs:postfunctions']", "view.transition.postfunctions");
    attachOnClickSafeEventTrigger(".status-properties", "view.status.properties");

    /**
     * @namespace JIRA.WorkflowDesigner.Analytics
     */
    return {
        /**
         * Reset the analytics object, exposed for testing.
         *
         * @private
         */
        _reset: function () {
            addedFirstStatus = false;
            addedFirstTransition = false;
        },

        /**
         * Triggers an event for toggling label visibility.
         *
         * @param {boolean} show Whether the labels were shown or hidden.
         */
        triggerToggleLabels: function(show) {
            triggerEvent("toggle.labels." + (show ? "on" : "off"));
        },

        /**
         * Triggers an event for adding a status
         */
        triggerAddStatus: function (permissions, analyticsData) {
            triggerEditEvent("add", "status", permissions, analyticsData);
        },

        /**
         * Triggers an event for updating a status
         */
        triggerUpdateStatus: function (permissions, analyticsData) {
            triggerEditEvent("update", "status", permissions, analyticsData);
        },

        /**
         * Triggers an event for adding a step
         */
        triggerAddStep: function (permissions, analyticsData) {
            triggerEditEvent("add", "step", permissions, analyticsData);
        },

        /**
         * Trigger an event for the very first time a user adds a status.
         *
         * @param {number} duration The time it took the user to add a status (in milliseconds).
         */
        triggerFirstAddStatus: function (duration, permissions) {
            var propertyKey = "jira.workflowdesigner.added.first.status",
                request = jQuery.Deferred();

            addedFirstStatus || (request = UserPropertiesAJAXManager.getProperty(propertyKey));
            request.done(function (property) {
                var shouldTrigger = !property.value;
                if (shouldTrigger) {
                    triggerEvent("first.add.status", {duration: duration}, permissions);
                    UserPropertiesAJAXManager.setProperty(propertyKey, true);
                }

                addedFirstStatus = true;
            });
        },

        /**
         * Triggers an event for updating a step
         */
        triggerUpdateStep: function (permissions, analyticsData) {
            triggerEditEvent("update", "step", permissions, analyticsData);
        },

        /**
         * Triggers an event for removing a step
         */
        triggerRemoveStep: function (permissions, analyticsData) {
            triggerEditEvent("remove", "step", permissions, analyticsData);
        },

        /**
         * Triggers an event for clicking link to remove a step
         */
        triggerRemoveStepClicked: function (permissions, analyticsData) {
            triggerEvent("status.remove.clicked", analyticsData, permissions);
        },

        triggerAddCommonTransition: function (permissions, analyticsData) {
            triggerEditEvent("add", "commontransition", permissions, analyticsData);
        },

        /**
         * Triggers an event for adding a transition
         */
        triggerAddTransition: function (permissions, analyticsData) {
            triggerEditEvent("add", "transition", permissions, analyticsData);
        },

        /**
         * Trigger an event for the very first time a user adds a transition.
         *
         * @param {number} duration The time it took the user to add a transition (in milliseconds).
         */
        triggerFirstAddTransition: function (duration, permissions, analyticsData) {
            var propertyKey = "jira.workflowdesigner.added.first.transition",
                request = jQuery.Deferred();

            addedFirstTransition || (request = UserPropertiesAJAXManager.getProperty(propertyKey));
            request.done(function (property) {
                var shouldTrigger = !property.value;
                if (shouldTrigger) {
                    triggerEvent("first.add.transition", jQuery.extend({duration: duration}, analyticsData), permissions);
                    UserPropertiesAJAXManager.setProperty(propertyKey, true);
                }

                addedFirstTransition = true;
            });
        },

        /**
         * Triggers an event for updating a transition
         */
        triggerUpdateTransition: function (permissions, analyticsData) {
            triggerEditEvent("update", "transition", permissions, analyticsData);
        },

        /**
         * Triggers an event for removing a transition
         */
        triggerRemoveTransition: function (permissions, analyticsData) {
            triggerEditEvent("remove", "transition", permissions, analyticsData);
        },

        /**
         * Triggers an event for adding a global transition
         */
        triggerAddGlobalTransition: function (analyticsData) {
            triggerEditEvent("add", "globaltransition", null, analyticsData);
        },

        /**
         * Triggers an event for updating a global transition
         */
        triggerUpdateGlobalTransition: function (permissions, analyticsData) {
            triggerEditEvent("update", "globaltransition", permissions, analyticsData);
        },

        /**
         * Triggers an event for removing a global transition
         */
        triggerRemoveGlobalTransition: function (permissions, analyticsData) {
            triggerEditEvent("remove", "globaltransition", permissions, analyticsData);
        },

        /**
         * Triggers an event for adding a looped transition
         */
        triggerAddLoopedTransition: function (analyticsData) {
            triggerEditEvent("add", "loopedtransition", null, analyticsData);
        },

        /**
         * Triggers an event for updating a looped transition
         */
        triggerUpdateLoopedTransition: function (permissions, analyticsData) {
            triggerEditEvent("update", "loopedtransition", permissions, analyticsData);
        },

        /**
         * Triggers an event for removing a looped transition
         */
        triggerRemoveLoopedTransition: function (analyticsData) {
            triggerEditEvent("remove", "loopedtransition", null, analyticsData);
        }
    };
});

AJS.namespace("JIRA.WorkflowDesigner.Analytics", null, require("workflow-designer/analytics"));
