AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer-loader",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var Analytics = require("workflow-designer/analytics");
    var UserPropertiesAJAXManager = require("workflow-designer/io/ajax/user-properties-ajax-manager");
    var _ = require("workflow-designer/underscore");
    var jQuery = require("jquery");
    var TestMocks = require("workflow-designer/test-mocks");

    module("Analytics", {
        setup: function () {
            var sandbox = this.sandbox = sinon.sandbox.create();

            if (AJS.Analytics) {
                this.originalAnalytics = AJS.Analytics;
            }
            else {
                this.originalAnalytics = null;
            }

            AJS.EventQueue = [];
            AJS.Analytics = {triggerPrivacyPolicySafeEvent: sinon.spy()};

            this.firstAddStepPropertyKey = "jira.workflowdesigner.added.first.status";
            this.firstAddTransitionPropertyKey = "jira.workflowdesigner.added.first.transition";
            this.getPropertyStub = sandbox.stub(UserPropertiesAJAXManager, "getProperty");
            this.setPropertyStub = sandbox.stub(UserPropertiesAJAXManager, "setProperty");
        },

        createPermissions: function (administrator, editPropertyOptions, selectScreenOnTransition, editWorkflow) {
            return {
                get: function (permission) {
                    if (permission === 'administrator') {
                        return administrator;
                    } else if (permission === 'editPropertyOptions') {
                        return editPropertyOptions;
                    } else if (permission === 'selectScreenOnTransition') {
                        return selectScreenOnTransition;
                    } else if (permission === 'editWorkflow') {
                        return editWorkflow;
                    }
                    return false;
                }
            };
        },

        teardown: function () {
            delete AJS.EventQueue;
            Analytics._reset();
            AJS.Analytics = this.originalAnalytics;
            this.sandbox.restore();
        }
    });

    test("Triggering events", function () {
        var tests = [
            ["triggerAddStatus", ["add", "status"]],
            ["triggerUpdateStatus", ["update", "status"]],

            ["triggerAddStep", ["add", "step"]],
            ["triggerUpdateStep", ["update", "step"]],
            ["triggerRemoveStep", ["remove", "step"]],

            ["triggerAddTransition", ["add", "transition"]],
            ["triggerUpdateTransition", ["update", "transition"]],
            ["triggerRemoveTransition", ["remove", "transition"]],

            ["triggerAddGlobalTransition", ["add", "globaltransition"]],
            ["triggerUpdateGlobalTransition", ["update", "globaltransition"]],
            ["triggerRemoveGlobalTransition", ["remove", "globaltransition"]]
        ];

        _.each(tests, function (test) {
            Analytics[test[0]]();

            var expectedEvent = TestMocks.analyticsEditEvent.apply(null, test[1]),
                actualEvent = _.last(AJS.EventQueue),
                msg = "The method " + test[0] + "() triggers an event with the expected values";

            deepEqual(actualEvent, expectedEvent, msg);
        }, this);
    });

    test("Triggering click events", function () {
        jQuery("<a class='enter-full-screen'>").appendTo("#qunit-fixture").click();
        ok(AJS.Analytics.triggerPrivacyPolicySafeEvent.calledWith("administration.workflow.fullscreen"), "A fullscreen event was triggered");

        var dataAttributesWithEventNames = [
            ["properties.key", "administration.workflow.view.transition.properties"],
            ["com.atlassian.jira.plugins.jira-transition-triggers-plugin:triggers-transitions-tab", "administration.workflow.view.transition.triggers-transitions-tab"],
            ["com.atlassian.jira.plugin.system.workfloweditor.transition.tabs:conditions", "administration.workflow.view.transition.conditions"],
            ["com.atlassian.jira.plugin.system.workfloweditor.transition.tabs:validators", "administration.workflow.view.transition.validators"],
            ["com.atlassian.jira.plugin.system.workfloweditor.transition.tabs:postfunctions", "administration.workflow.view.transition.postfunctions"]
        ];

        _.each(dataAttributesWithEventNames, function (test) {
            jQuery("<a data-transition-option-key='" + test[0] + "'>").appendTo("#qunit-fixture").click();
            ok(AJS.Analytics.triggerPrivacyPolicySafeEvent.calledWith(test[1]), "Event with expected name was triggered: " + test[1]);
        });
    });

    test("Triggering toggle labels event", function () {
        Analytics.triggerToggleLabels(true);
        equal(_.last(AJS.EventQueue).name, "administration.workflow.toggle.labels.on", "Toggle labels on event was triggered");

        Analytics.triggerToggleLabels(false);
        equal(_.last(AJS.EventQueue).name, "administration.workflow.toggle.labels.off", "Toggle labels off event was triggered");
    });

    test("Subsequent triggerFirstAddStatus() calls don't make AJAX requests", function () {
        var getPropertyResult = {
            key: this.firstAddStepPropertyKey,
            value: true
        };

        this.getPropertyStub.returns(jQuery.Deferred().resolve(getPropertyResult));
        Analytics.triggerFirstAddStatus(42);
        Analytics.triggerFirstAddStatus(42);

        equal(this.getPropertyStub.callCount, 1, "UserPropertiesAJAXManager#getProperty was only called once");
    });

    test("triggerFirstAddStatus()", function () {
        var getPropertyDeferred = jQuery.Deferred();

        this.getPropertyStub.returns(getPropertyDeferred);
        Analytics.triggerFirstAddStatus(42);

        equal(this.getPropertyStub.callCount, 1, "UserPropertiesAJAXManager#getProperty was called");
        deepEqual(this.getPropertyStub.args[0], [this.firstAddStepPropertyKey], "It was passed the correct argument");

        getPropertyDeferred.resolve({key: this.firstAddStepPropertyKey, value: null});
        equal(_.last(AJS.EventQueue).name, "administration.workflow.first.add.status", "A first add step event was triggered");
        deepEqual(_.last(AJS.EventQueue).properties, {duration: 42}, "It has the correct properties");
        equal(this.setPropertyStub.callCount, 1, "UserPropertiesAJAXManager#setProperty was called");
        deepEqual(this.setPropertyStub.args[0], [this.firstAddStepPropertyKey, true], "It was passed the correct arguments");
    });

    test("triggerFirstAddStatus() fails silently", function () {
        var getPropertyDeferred = jQuery.Deferred().reject();

        this.getPropertyStub.returns(getPropertyDeferred);
        Analytics.triggerFirstAddStatus(42);

        equal(AJS.EventQueue.length, 0, "No analytics event was triggered");
        equal(this.setPropertyStub.callCount, 0, "UserPropertiesAJAXManager#setProperty() wasn't called");
    });

    test("triggerFirstAddStatus() only triggers an event for the user's first status", function () {
        var getPropertyResult = {
            key: this.firstAddStepPropertyKey,
            value: true
        };

        this.getPropertyStub.returns(jQuery.Deferred().resolve(getPropertyResult));
        Analytics.triggerFirstAddStatus(42);

        equal(AJS.EventQueue.length, 0, "No analytics event was triggered");
        equal(this.setPropertyStub.callCount, 0, "UserPropertiesAJAXManager#setProperty wasn't called");
    });

    test("Subsequent triggerFirstAddTransition() calls don't make AJAX requests", function () {
        var getPropertyResult = {
            key: this.firstAddTransitionPropertyKey,
            value: true
        };

        this.getPropertyStub.returns(jQuery.Deferred().resolve(getPropertyResult));
        Analytics.triggerFirstAddTransition(42);
        Analytics.triggerFirstAddTransition(42);

        equal(this.getPropertyStub.callCount, 1, "UserPropertiesAJAXManager#getProperty was only called once");
    });

    test("triggerFirstAddTransition()", function () {
        var getPropertyDeferred = jQuery.Deferred();

        this.getPropertyStub.returns(getPropertyDeferred);
        Analytics.triggerFirstAddTransition(42);

        equal(this.getPropertyStub.callCount, 1, "UserPropertiesAJAXManager#getProperty was called");
        deepEqual(this.getPropertyStub.args[0], [this.firstAddTransitionPropertyKey], "It was passed the correct argument");

        getPropertyDeferred.resolve({key: this.firstAddTransitionPropertyKey, value: null});

        equal(_.last(AJS.EventQueue).name, "administration.workflow.first.add.transition", "A first add transition event was triggered");
        deepEqual(_.last(AJS.EventQueue).properties, {duration: 42}, "It has the correct properties");
        equal(this.setPropertyStub.callCount, 1, "UserPropertiesAJAXManager#setProperty was called");
        deepEqual(this.setPropertyStub.args[0], [this.firstAddTransitionPropertyKey, true], "It was passed the correct arguments");
    });

    test("triggerFirstAddStatus() fails silently", function () {
        var getPropertyDeferred = jQuery.Deferred().reject();

        this.getPropertyStub.returns(getPropertyDeferred);
        Analytics.triggerFirstAddTransition(42);

        equal(AJS.EventQueue.length, 0, "No analytics event was triggered");
        equal(this.setPropertyStub.callCount, 0, "UserPropertiesAJAXManager#setProperty() wasn't called");
    });

    test("triggerFirstAddStatus() only triggers an event for the user's first status", function () {
        var getPropertyResult = {
            key: this.firstAddTransitionPropertyKey,
            value: true
        };

        this.getPropertyStub.returns(jQuery.Deferred().resolve(getPropertyResult));
        Analytics.triggerFirstAddTransition(42);

        equal(AJS.EventQueue.length, 0, "No analytics event was triggered");
        equal(this.setPropertyStub.callCount, 0, "UserPropertiesAJAXManager#setProperty wasn't called");
    });

    test("additional permissions with editWorkflow affects analytics key", function () {
        var permissions = this.createPermissions(false, false, false, true);

        Analytics.triggerAddStatus(permissions);

        equal(AJS.EventQueue.length, 1, "One analytics event should be triggered");
        deepEqual(_.last(AJS.EventQueue), {
            name: "administration.workflow.editworkflowpermission.edit",
            properties: {
                mode: "new-diagram",
                action: "add",
                object: "status"
            }
        }, "Correct event is produced");
    });

    test("simultanous administrator and editWorkflow permissions should not affect analytics key", function () {
        var permissions = this.createPermissions(true, false, false, true);

        Analytics.triggerAddStatus(permissions);

        equal(AJS.EventQueue.length, 1, "One analytics event should be triggered");
        deepEqual(_.last(AJS.EventQueue), {
            name: "administration.workflow.edit",
            properties: {
                mode: "new-diagram",
                action: "add",
                object: "status"
            }
        }, "Correct event is produced");
    });
    
});
