define("workflow-designer/layout/locator/manhattan-percentage-locator", [
    "workflow-designer/vector-2d",
    "workflow-designer/draw-2d",
    "workflow-designer/underscore"
], function(
    Vector2D,
    draw2d,
    _
) {
    /**
     * @param {object} lineSegment A {@link draw2d.shape.basic.PolyLine} line segment.
     * @returns {number} The length of `lineSegment`.
     */
    function getLineSegmentLength(lineSegment) {
        return new Vector2D(lineSegment.start, lineSegment.end).getLength();
    }

    /**
     * @param {draw2d.shape.basic.PolyLine} polyline A Draw2D polyline.
     * @returns {number} The length of `polyline`
     */
    function getPolylineLength(polyline) {
        var lineSegments = polyline.getSegments().asArray();
        return _.reduce(lineSegments, function (length, lineSegment) {
            return length + getLineSegmentLength(lineSegment);
        }, 0);
    }

    return draw2d.layout.locator.ConnectionLocator.extend(
    /** @lends JIRA.WorkflowDesigner.Layout.Locator.ManhattanPercentageLocator# */
    {
        /**
         * @classdesc A connection locator that positions the target a percentage along the connection.
         * @constructs
         * @extends draw2d.layout.locator.ConnectionLocator
         * @param {draw2d.Connection} parent The locator's parent connection.
         * @param {number} percentage The percentage along the connection to position the target (`0 <= percentage <= 1`).
         */
        init: function (parent, percentage) {
            this._percentage = percentage;
            this._super(parent);
        },

        /**
         * Position a figure on the parent connection.
         *
         * @param {number} index The child index of `target`.
         * @param {draw2d.Figure} target The figure to position.
         */
        relocate: function (index, target) {
            var connection = this.getParent(),
                distance = getPolylineLength(connection) * this._percentage,
                lineSegment,
                offset;

            // Find the line segment on which `target` should be positioned.
            lineSegment = _.find(connection.getSegments().asArray(), function (lineSegment) {
                var length = getLineSegmentLength(lineSegment);
                if (length >= distance) {
                    return true;
                }

                distance -= length;
            });

            if (lineSegment) {
                offset = new Vector2D(lineSegment.start, lineSegment.end);
                offset.normalize().multiply(distance).translate(target.getWidth() / -2, target.getHeight() / -2);
                target.setPosition(lineSegment.start.getTranslated(offset));
            }
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.Layout.Locator.ManhattanPercentageLocator", null, require("workflow-designer/layout/locator/manhattan-percentage-locator"));