define("workflow-designer/last-saved-by/controller", [
    "workflow-designer/backbone",
    "workflow-designer/last-saved-by/model",
    "workflow-designer/last-saved-by/view",
    "jira/util/users/logged-in-user"
], function(
    Backbone,
    LastSavedByModel,
    LastSavedByView,
    loggedInUser
) {
    "use strict";

    /*jshint -W104 */ // W104 is 'const' is only available in JavaScript 1.7.
    /**
     * After this time "pending" state will be changed to "pending-slow"
     * @type {number}
     */
    const PENDING_SLOW_DELAY = 600;

    /**
     * Currently logged in user data in the form of `workflowModel.updateAuthor`
     * @type {{displayName: '', userName: ''}}
     */
    const CURRENT_USER = {
        displayName: loggedInUser.fullName(),
        userName: loggedInUser.username()
    };
    /*jshint +W104 */

    return Backbone.Marionette.Controller.extend({
        /**
         *
         * @param {module:workflow-designer/internal-api/save-status-service} options.saveStatusService handles a state of save status
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's `WorkflowModel`.
         * @param {Marionette.Region} options.region The region to render within.
         */
        initialize: function (options) {
            this._workflowModel = options.workflowModel;
            this.region = options.region;

            this.model = new LastSavedByModel();
            this.view = new LastSavedByView({
                model: this.model
            });

            this.model.set("currentUserName", CURRENT_USER.userName);

            this.listenTo(this._workflowModel, "change:updateAuthor change:updatedDate", this._workflowModelChanged);

            var saveStatusService = options.saveStatusService;
            this.listenTo(saveStatusService, "state:pending", this._pendingStateHandler);
            this.listenTo(saveStatusService, "state:success", this._successStateHandler);
            this.listenTo(saveStatusService, "state:failure", this._failureStateHandler);
        },

        show: function() {
            this.region.show(this.view);
        },

        onClose: function() {
            window.clearTimeout(this._pendingStateTimeout);
        },

        _workflowModelChanged: function() {
            if (this._workflowModel.get("updateAuthor") && this._workflowModel.get("updatedDate")) {
                this.stopListening(this._workflowModel);
                this.model.set({
                    updateAuthor: this._workflowModel.get("updateAuthor"),
                    updatedDate: this._workflowModel.get("updatedDate")
                });
            }
        },

        _pendingStateHandler: function() {
            this.model.set({
                state: "pending",
                icon: "wait"
            });

            if (!this._pendingStateTimeout) {
                this._pendingStateTimeout = window.setTimeout(function() {
                    delete this._pendingStateTimeout;

                    this.model.set({
                        state: "pending-slow"
                    });
                }.bind(this), PENDING_SLOW_DELAY);
            }
        },

        _successStateHandler: function(lastSyncTime) {
            window.clearTimeout(this._pendingStateTimeout);
            delete this._pendingStateTimeout;

            var icon = this.model.get("state") === "failure" ? this.model.get("icon") : "approve";
            this.model.set({
                state: "success",
                icon: icon,
                updatedDate: lastSyncTime,
                updateAuthor: CURRENT_USER
            });
        },

        _failureStateHandler: function(lastSyncTime) {
            window.clearTimeout(this._pendingStateTimeout);
            delete this._pendingStateTimeout;

            this.model.set({
                state: "failure",
                icon: "error",
                updatedDate: lastSyncTime,
                updateAuthor: CURRENT_USER
            });
        }
    });
});
