AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer-last-saved-by"
], function () {

    /*jshint -W104 */ // W104 is 'const' is only available in JavaScript 1.7.
    // look at LastSavedByController.js#PENDING_SLOW_DELAY for the description
    // +10 ms is added to avoid flakiness
    const PENDING_SLOW_DELAY = 600 + 10;
    /*jshint +W104 */

    var Backbone = require("workflow-designer/backbone");
    var _ = require("workflow-designer/underscore");

    module("LastSavedByController", {
        setup: function () {
            this.sandbox = sinon.sandbox.create();
            this.context = AJS.test.mockableModuleContext();

            this.mockLoggedInUser();
            this.mockView();

            this.workflowModel = this.mockWorkflowModel();
            this.region = this.mockRegion();
            this.saveStatusService = this.mockSaveStatusService();

            var Controller = this.context.require("workflow-designer/last-saved-by/controller");
            this.sandbox.spy(Controller.prototype, "_workflowModelChanged");
            this.sandbox.spy(Controller.prototype, "_pendingStateHandler");
            this.sandbox.spy(Controller.prototype, "_successStateHandler");
            this.sandbox.spy(Controller.prototype, "_failureStateHandler");

            this.controller = new Controller({
                workflowModel: this.workflowModel,
                region: this.region,
                saveStatusService: this.saveStatusService
            });
        },
        teardown: function () {
            this.sandbox.restore();
        },
        mockLoggedInUser: function () {
            this.context.mock("jira/util/users/logged-in-user", {
                fullName: function() { return "Bob Dude"; },
                username: function() { return "bob"; }
            });
        },
        mockWorkflowModel: function (options) {
            var WorkflowModel = Backbone.Model.extend({
                updateAuthor: null,
                updatedDate: null
            });
            return new WorkflowModel(_.clone(options || {}));
        },
        mockView: function () {
            this.context.mock("workflow-designer/last-saved-by/view", sinon.stub());
        },
        mockRegion: function () {
            return sinon.createStubInstance(Backbone.Marionette.Region);
        },
        mockSaveStatusService: function () {
            return _.extend({}, Backbone.Events);
        },
        getCurrentUser: function () {
            var loggedInUser = this.context.require("jira/util/users/logged-in-user");
            return {
                displayName: loggedInUser.fullName(),
                userName: loggedInUser.username()
            };
        }
    });

    test("Should react on `updateAuthor` change of workflow model", function () {
        sinon.assert.notCalled(this.controller._workflowModelChanged);

        this.workflowModel.set({
            updateAuthor: {}
        });

        sinon.assert.calledOnce(this.controller._workflowModelChanged);
    });

    test("Should react on `updatedDate` change of workflow model", function () {
        sinon.assert.notCalled(this.controller._workflowModelChanged);

        this.workflowModel.set({
            updatedDate: new Date().getTime()
        });

        sinon.assert.calledOnce(this.controller._workflowModelChanged);
    });

    test("Should update a model and stop listening on workflow model when both updateAuthor and updatedDate were set", function () {
        var author = {displayName: 'JJ', userName: 'JJadmin'};
        var datetime = new Date().getTime();

        this.workflowModel.set({
            updateAuthor: author,
            updatedDate: datetime
        });

        var callCount = this.controller._workflowModelChanged.callCount;

        strictEqual(this.controller.model.get('updateAuthor'), author);
        strictEqual(this.controller.model.get('updatedDate'), datetime);

        this.workflowModel.set({
            updateAuthor: {},
            updatedDate: new Date().getTime()
        });

        equal(this.controller._workflowModelChanged.callCount, callCount, "Should stop listening on workflow model");
    });

    test("Should show LastSavedByView within provided region", function () {
        sinon.assert.notCalled(this.region.show);

        this.controller.show();
        sinon.assert.calledWith(this.region.show, this.controller.view);
    });

    test("Should set `currentUserName` to currently logged in user", function () {
        var username = this.context.require("jira/util/users/logged-in-user").username();
        strictEqual(this.controller.model.get("currentUserName"), username);
    });

    test("Should react on `state:pending` event", function () {
        sinon.assert.notCalled(this.controller._pendingStateHandler);

        this.saveStatusService.trigger("state:pending");
        sinon.assert.calledOnce(this.controller._pendingStateHandler);

        this.saveStatusService.trigger("state:pending");
        sinon.assert.calledTwice(this.controller._pendingStateHandler);
    });
    
    test("Should react on `state:success` event", function () {
        sinon.assert.notCalled(this.controller._successStateHandler);

        this.saveStatusService.trigger("state:success");
        sinon.assert.calledOnce(this.controller._successStateHandler);

        this.saveStatusService.trigger("state:success");
        sinon.assert.calledTwice(this.controller._successStateHandler);
    });

    test("Should react on `state:failure` event", function () {
        sinon.assert.notCalled(this.controller._failureStateHandler);

        this.saveStatusService.trigger("state:failure");
        sinon.assert.calledOnce(this.controller._failureStateHandler);

        this.saveStatusService.trigger("state:failure");
        sinon.assert.calledTwice(this.controller._failureStateHandler);
    });

    test("Should properly update a model on `state:success` when current `state` is not `failure`", function () {
        var datetime = new Date().getTime();

        this.controller.model.set("state", "pending");

        this.saveStatusService.trigger("state:success", datetime);

        equal(this.controller.model.get("state"), "success", "incorrect state");
        equal(this.controller.model.get("icon"), "approve", "incorrect icon");
        equal(this.controller.model.get("updatedDate"), datetime, "incorrect updatedDate");
        deepEqual(this.controller.model.get("updateAuthor"), this.getCurrentUser(), "incorrect updateAuthor");
    });

    test("Should properly update a model on `state:success` when current `state` is `failure`", function () {
        var datetime = new Date().getTime();

        this.controller.model.set("state", "failure");
        this.controller.model.set("icon", "error");
        this.saveStatusService.trigger("state:success", datetime);

        equal(this.controller.model.get("state"), "success", "incorrect state");
        equal(this.controller.model.get("icon"), "error", "incorrect icon");
        equal(this.controller.model.get("updatedDate"), datetime, "incorrect updatedDate");
        deepEqual(this.controller.model.get("updateAuthor"), this.getCurrentUser(), "incorrect updateAuthor");
    });

    test("Should properly update a model on `state:failure`", function () {
        var datetime = new Date().getTime();

        notEqual(this.controller.model.get("state"), "failure");

        this.saveStatusService.trigger("state:failure", datetime);

        equal(this.controller.model.get("state"), "failure", "incorrect state");
        equal(this.controller.model.get("icon"), "error", "incorrect icon");
        equal(this.controller.model.get("updatedDate"), datetime, "incorrect updatedDate");
        deepEqual(this.controller.model.get("updateAuthor"), this.getCurrentUser(), "incorrect updateAuthor");
    });

    test("Should properly update a model on `state:pending`", function () {
        notEqual(this.controller.model.get("state"), "pending");

        this.saveStatusService.trigger("state:pending");

        equal(this.controller.model.get("state"), "pending", "incorrect state");
        equal(this.controller.model.get("icon"), "wait", "incorrect icon");
    });

    test("Should change a state from `pending` to `pending-slow` callback after some time", function () {
        this.sandbox.useFakeTimers();

        this.saveStatusService.trigger("state:pending");

        this.sandbox.clock.tick(PENDING_SLOW_DELAY);

        equal(this.controller.model.get("state"), "pending-slow");
    });

    test("Should not handle `pending-slow` callback after closing the controller", function () {
        this.sandbox.useFakeTimers();

        this.saveStatusService.trigger("state:pending");
        this.controller.close();

        this.sandbox.clock.tick(PENDING_SLOW_DELAY);

        notEqual(this.controller.model.get("state"), "pending-slow", "`state` should remain `pending`");
    });
    
    test("Should not handle `pending-slow` callback after receiving `state:success`", function () {
        this.sandbox.useFakeTimers();

        this.saveStatusService.trigger("state:pending");
        this.sandbox.clock.tick(PENDING_SLOW_DELAY / 2);

        this.saveStatusService.trigger("state:success", new Date().getTime());
        this.sandbox.clock.tick(PENDING_SLOW_DELAY / 2);

        notEqual(this.controller.model.get("state"), "pending-slow", "`state` should remain `pending`");
    });

    test("Should not handle `pending-slow` callback after receiving `state:failure`", function () {
        this.sandbox.useFakeTimers();

        this.saveStatusService.trigger("state:pending");
        this.sandbox.clock.tick(PENDING_SLOW_DELAY / 2);

        this.saveStatusService.trigger("state:failure", new Date().getTime());
        this.sandbox.clock.tick(PENDING_SLOW_DELAY / 2);

        notEqual(this.controller.model.get("state"), "pending-slow", "`state` should remain `pending`");
    });

    test("Should not handle `pending-slow` callback twice after receiving multiple `state:pending`", function () {
        this.sandbox.useFakeTimers();
        this.sandbox.stub(this.controller.model, "set");
        var pendingSlowSpy = this.controller.model.set.withArgs({state: "pending-slow"});

        sinon.assert.notCalled(pendingSlowSpy);

        this.saveStatusService.trigger("state:pending");
        this.sandbox.clock.tick(10);

        this.saveStatusService.trigger("state:pending");
        this.sandbox.clock.tick(10);

        sinon.assert.notCalled(pendingSlowSpy);

        this.saveStatusService.trigger("state:pending");
        this.sandbox.clock.tick(PENDING_SLOW_DELAY);

        sinon.assert.calledOnce(pendingSlowSpy);

        this.sandbox.clock.tick(PENDING_SLOW_DELAY);

        sinon.assert.calledOnce(pendingSlowSpy);
    });
});
