define("workflow-designer/io/ajax/workflow-statuses-ajax-manager", [
    "workflow-designer/io/ajax/ajax-manager",
    "workflow-designer/io/workflow-data-reader",
    "workflow-designer/io/ajax/statuses-ajax-manager",
    "workflow-designer/io/error-handler",
    "workflow-designer/underscore",
    "jquery",
    "wrm/context-path"
], function(
    AJAXManager,
    WorkflowDataReader,
    StatusesAJAXManager,
    ErrorHandler,
    _,
    jQuery,
    contextPath
) {
    /**
     * Cache for statuses
     *
     * @inner
     * @type {Object|undefined}
     */
    var statuses;

    /**
     * Handles manipulating a workflow's statuses.
     *
     * @namespace
     * @extends JIRA.WorkflowDesigner.IO.AJAX.AJAXManager
     */
    return new (AJAXManager.extend(/** @lends JIRA.WorkflowDesigner.IO.AJAX.WorkflowStatusesAJAXManager */{
        /**
         * Add a status to a workflow.
         *
         * @param {object} options
         * @param {boolean} options.createGlobalTransition Whether a global transition to the status should also be created.
         * @param {string} options.statusId The ID of the status to add.
         * @param {string} options.workflowName The workflow's name.
         * @return {jQuery.Deferred} A deferred that is resolved on success.
         */
        addStatus: function (options) {
            var deferred = jQuery.Deferred(),
                request;

            request = this._makeRequest({
                data: _.pick(options, "createGlobalTransition", "statusId", "workflowName"),
                dataType: "json",
                type: "post",
                url: contextPath() + "/rest/workflowDesigner/latest/workflows/statuses"
            });

            request.done(function (response) {
                deferred.resolve(WorkflowDataReader.read(response));
            });

            request.fail(function (response, status) {
                deferred.reject(ErrorHandler.getErrorMessage(response, status));
            });

            return deferred.promise();
        },

        /**
         * Create a new status and add it to a workflow.
         *
         * @param {object} options
         * @param {string} options.description The status's description.
         * @param {string} options.name The status's name.
         * @param {string} options.statusCategoryId The ID of the status category to use.
         * @param {string} options.workflowName The workflow's name.
         * @return {jQuery.Deferred} A deferred that is resolved on success.
         */
        createStatus: function (options) {
            var deferred = jQuery.Deferred(),
                request;

            request = this._makeRequest({
                data: _.pick(options, "createGlobalTransition", "description", "name", "statusCategoryId", "workflowName"),
                dataType: "json",
                type: "post",
                url: contextPath() + "/rest/workflowDesigner/latest/workflows/statuses/create"
            });

            request.done(function (response) {
                deferred.resolve(WorkflowDataReader.read(response));
                StatusesAJAXManager.reset();
                this.reset();
            }.bind(this));

            request.fail(function (response, status) {
                deferred.reject(ErrorHandler.getErrorMessage(response, status));
            });

            return deferred.promise();
        },

        /**
         * Remove a status from the workflow.
         *
         * @param {object} options
         * @param {string} options.statusId The status's identifier.
         * @param {string} options.workflowName The workflow's name.
         * @return {jQuery.Deferred} A deferred that is resolved on success.
         */
        removeStatus: function (options) {
            var deferred = jQuery.Deferred(),
                request;

            request = this._makeRequest({
                data: _.pick(options, "statusId", "workflowName"),
                dataType: "json",
                type: "delete",
                url: contextPath() + "/rest/workflowDesigner/latest/workflows/statuses"
            });

            request.done(function (response) {
                deferred.resolve(WorkflowDataReader.read(response));
            });

            request.fail(function (response, status) {
                deferred.reject(ErrorHandler.getErrorMessage(response, status));
            });

            return deferred.promise();
        },

        /**
         * Validate status removal from the workflow.
         *
         * @param {object} options
         * @param {string} options.statusId The status's identifier.
         * @param {string} options.workflowName The workflow's name.
         * @return {jQuery.Deferred} A deferred that is resolved on success.
         */
        validateRemoveStatus: function (options) {
            var deferred = jQuery.Deferred(),
                request;

            request = this._makeRequest({
                data: _.pick(options, "statusId", "workflowName"),
                dataType: "json",
                type: "post",
                url: contextPath() + "/rest/workflowDesigner/latest/workflows/statuses/validateRemove"
            }, true);

            request.done(function (response) {
                deferred.resolve(WorkflowDataReader.read(response));
            });

            request.fail(function (response, status) {
                deferred.reject(ErrorHandler.getErrorMessage(response, status));
            });

            return deferred.promise();
        },

        /**
         * Edit an existing status that is present on the workflow.
         *
         * @param {object} options
         * @param {string} options.statusId The status's id.
         * @param {string} options.description The status's description.
         * @param {string} options.name The status's name.
         * @param {string} options.workflowName The workflow's name.
         * @return {jQuery.Deferred} A deferred that is resolved on success.
         */
        updateStatus: function (options) {
            var deferred = jQuery.Deferred(),
                request;

            request = this._makeRequest({
                data: _.pick(options, "description", "name", "statusCategoryId", "statusId", "workflowName"),
                dataType: "json",
                type: "put",
                url: contextPath() + "/rest/workflowDesigner/latest/workflows/statuses"
            });

            request.done(deferred.resolve);

            request.fail(function (response, status) {
                deferred.reject(ErrorHandler.getErrorMessage(response, status));
            });

            return deferred.promise();
        },

        /**
         * Retrieve information about all statuses accessible in context of the specified workflow.
         *
         * The response is cached under the assumption that statuses don't change often.
         *
         * @return {jQuery.Deferred} A deferred that is resolved with status data.
         */
        getStatuses: function (options) {
            var deferred = jQuery.Deferred();

            if (statuses) {
                deferred.resolve(statuses);
            } else {
                var request = this._makeRequest({
                    data: _.pick(options, "workflowName"),
                    dataType: "json",
                    url: contextPath() + "/rest/workflowDesigner/latest/workflows/statuses"
                });

                request.done(function (response) {
                    statuses = response;
                    deferred.resolve(statuses);
                });

                request.fail(function (response, status) {
                    deferred.reject(ErrorHandler.getErrorMessage(response, status));
                });
            }

            return deferred.promise();
        },

        /**
         * Reset the manager, clearing its cache.
         */
        reset: function () {
            statuses = undefined;
        }
    }))();
});

AJS.namespace("JIRA.WorkflowDesigner.IO.AJAX.WorkflowStatusesAJAXManager", null, require("workflow-designer/io/ajax/workflow-statuses-ajax-manager"));