AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer-io",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var WorkflowStatusesAJAXManager = require('workflow-designer/io/ajax/workflow-statuses-ajax-manager');
    var StatusesAJAXManager = require('workflow-designer/io/ajax/statuses-ajax-manager');
    var WorkflowDataReader = require("workflow-designer/io/workflow-data-reader");
    var TestUtilities = require("workflow-designer/test-utilities");
    var contextPath = require("wrm/context-path");

    module("WorkflowStatusesAJAXManager", {
        setup: function () {
            this.addStatusURL = contextPath() + "/rest/workflowDesigner/latest/workflows/statuses";
            this.createStatusURL = contextPath() + "/rest/workflowDesigner/latest/workflows/statuses/create";
            this.sandbox = sinon.sandbox.create({useFakeServer: true});
            this.workflowDataReaderStub = this.sandbox.stub(WorkflowDataReader, "read");
            this.workflowStatusesAJAXManager = WorkflowStatusesAJAXManager;
            this.triggerStub = this.sandbox.stub(this.workflowStatusesAJAXManager, "trigger");
        },

        teardown: function () {
            this.sandbox.restore();
            this.workflowStatusesAJAXManager.reset();
        }
    });

    test("addStatus()", function () {
        var deferred,
            sentRequestBody,
            expectedRequestBody,
            response = {};

        sinon.assert.notCalled(this.triggerStub);

        deferred = this.workflowStatusesAJAXManager.addStatus({
            createGlobalTransition: true,
            statusId: "Open",
            workflowName: "Workflow"
        });

        sinon.assert.calledOnce(this.triggerStub);

        this.workflowDataReaderStub.returns(response);
        this.sandbox.server.respondWith(this.addStatusURL, [200, {}, "{}"]);
        this.sandbox.server.respond();

        expectedRequestBody = {
            createGlobalTransition: "true",
            statusId: "Open",
            workflowName: "Workflow"
        };
        sentRequestBody = TestUtilities.getRequestBodyAsObject(this.sandbox.server.requests[0].requestBody);
        deepEqual(sentRequestBody, expectedRequestBody, "The request's body was correct");

        equal(this.workflowDataReaderStub.callCount, 1, "The AJAX response was passed to WorkflowDataReader.read()");
        deepEqual(this.workflowDataReaderStub.args[0], [{}], "It was passed the correct arguments");

        equal(deferred.state(), "resolved", "The returned deferred was resolved.");
        deferred.done(function (data) {
            ok(data === response, "It was resolved with the output of WorkflowDataReader.read()");
        });
    });

    test("addStatus() rejects the returned deferred on error", function () {
        var deferred;

        sinon.assert.notCalled(this.triggerStub);

        deferred = this.workflowStatusesAJAXManager.addStatus({
            createGlobalTransition: true,
            statusId: "Closed",
            workflowName: "Workflow"
        });

        sinon.assert.calledOnce(this.triggerStub);

        this.sandbox.server.respondWith(this.addStatusURL, [400, {}, "No deal."]);
        this.sandbox.server.respond();

        equal(deferred.state(), "rejected", "The returned deferred was rejected");
        deferred.fail(function (errorMessage) {
            equal(errorMessage, "No deal.", "It was rejected with the error message.");
        });
    });

    test("createStatus() with create global transition flag", function () {
        var deferred,
            expectedRequestBody,
            response = {},
            sentRequestBody,
            statusesAJAXManagerResetSpy = this.sandbox.spy(StatusesAJAXManager, "reset"),
            workflowStatusesAJAXManagerResetSpy = this.sandbox.spy(WorkflowStatusesAJAXManager, "reset");

        this.sandbox.server.respondWith(contextPath() + "/rest/workflowDesigner/latest/workflows/statuses", [200, {}, ""]);
        this.sandbox.server.respond();

        sinon.assert.notCalled(this.triggerStub);

        deferred = this.workflowStatusesAJAXManager.createStatus({
            createGlobalTransition: true,
            description: "Description",
            name: "Status",
            statusCategoryId: "1",
            workflowName: "Workflow"
        });

        sinon.assert.calledOnce(this.triggerStub);

        this.workflowDataReaderStub.returns(response);
        this.sandbox.server.respondWith(this.createStatusURL, [200, {}, "{}"]);
        this.sandbox.server.respond();

        sentRequestBody = TestUtilities.getRequestBodyAsObject(this.sandbox.server.requests[0].requestBody);
        expectedRequestBody = {
            createGlobalTransition: "true",
            description: "Description",
            name: "Status",
            statusCategoryId: "1",
            workflowName: "Workflow"
        };
        deepEqual(sentRequestBody, expectedRequestBody, "The request body for create was correct.");

        equal(this.workflowDataReaderStub.callCount, 1, "The AJAX response was passed to WorkflowDataReader.read()");
        deepEqual(this.workflowDataReaderStub.args[0], [{}], "It was passed the correct arguments");

        equal(deferred.state(), "resolved", "The returned deferred was resolved.");
        equal(statusesAJAXManagerResetSpy.callCount, 1, "StatusesAJAXManager.reset() was called");
        equal(workflowStatusesAJAXManagerResetSpy.callCount, 1, "WorkflowStatusesAJAXManagerResetSpy.reset() was called");

        deferred.done(function (data) {
            ok(data === response, "It was resolved with the output of WorkflowDataReader.read()");
        });
    });

    test("createStatus() rejects the returned deferred on error", function () {
        var deferred;

        sinon.assert.notCalled(this.triggerStub);

        deferred = this.workflowStatusesAJAXManager.createStatus({
            description: "A plain old status.",
            name: "Status",
            workflowName: "Workflow"
        });

        sinon.assert.calledOnce(this.triggerStub);

        this.sandbox.server.respondWith(this.createStatusURL, [400, {}, "No"]);
        this.sandbox.server.respond();

        equal(deferred.state(), "rejected", "The returned deferred was rejected");
        deferred.fail(function (errorMessage) {
            equal(errorMessage, "No", "It was rejected with the error message");
        });
    });

    test("removeStatus() resolves deferred result on success and has correct request body", function () {
        var deferred,
            expectedRequestBody,
            sentRequestBody;

        sinon.assert.notCalled(this.triggerStub);

        deferred = this.workflowStatusesAJAXManager.removeStatus({
            statusId: 100,
            workflowName: "Workflow"
        });

        sinon.assert.calledOnce(this.triggerStub);

        this.sandbox.server.respondWith(contextPath() + "/rest/workflowDesigner/latest/workflows/statuses", [200, {}, ""]);
        this.sandbox.server.respond();

        expectedRequestBody = {
            statusId: "100",
            workflowName: "Workflow"
        };

        sentRequestBody = TestUtilities.getRequestBodyAsObject(this.sandbox.server.requests[0].requestBody);
        deepEqual(sentRequestBody, expectedRequestBody, "The request body matches the arguments sent to identify a status for removal");

        equal(deferred.state(), "resolved", "The returned deferred was resolved.");
    });

    test("removeStatus() rejects the returned deferred on error", function () {
        var deferred;

        sinon.assert.notCalled(this.triggerStub);

        deferred = this.workflowStatusesAJAXManager.removeStatus({
            statusId: 100,
            workflowName: "Workflow"
        });

        sinon.assert.calledOnce(this.triggerStub);

        TestUtilities.respondWithErrorAndAssertRejected(
            deferred, "/rest/workflowDesigner/latest/workflows/statuses", this.sandbox.server);
    });

    test("updateStatus() resolves deferred result on success", function () {
        var actualRequestObject,
            deferred,
            expectedRequestObject;

        expectedRequestObject = {
            description: "New description!",
            name: "New Name",
            statusCategoryId: "1",
            statusId: "100",
            workflowName: "Workflow"
        };

        sinon.assert.notCalled(this.triggerStub);

        deferred = this.workflowStatusesAJAXManager.updateStatus({
            description: "New description!",
            name: "New Name",
            someUnknownParameter: true, // Shouldn't be sent
            statusCategoryId: "1",
            statusId: "100",
            workflowName: "Workflow"
        });

        sinon.assert.calledOnce(this.triggerStub);

        this.sandbox.server.respondWith(contextPath() + "/rest/workflowDesigner/latest/workflows/statuses", [200, {}, ""]);
        this.sandbox.server.respond();

        actualRequestObject = TestUtilities.getRequestBodyAsObject(this.sandbox.server.requests[0].requestBody);
        deepEqual(actualRequestObject, expectedRequestObject, "The request body for update was correct.");
        equal(deferred.state(), "resolved", "The returned deferred was resolved.");
    });

    test("updateStatus() rejects the returned deferred on error", function () {
        sinon.assert.notCalled(this.triggerStub);

        var deferred = this.workflowStatusesAJAXManager.updateStatus({
            transitionId: 100,
            name: "name",
            description: "description",
            workflowName: "Workflow"
        });

        sinon.assert.calledOnce(this.triggerStub);

        TestUtilities.respondWithErrorAndAssertRejected(
            deferred, "/rest/workflowDesigner/latest/workflows/statuses", this.sandbox.server);
    });

    test("getStatuses()", function () {
        sinon.assert.notCalled(this.triggerStub);

        var deferred = this.workflowStatusesAJAXManager.getStatuses({workflowName: 'workflow name'}),
            response = [{name: "Open"}, {name: "Closed"}],
            URL = contextPath() + "/rest/workflowDesigner/latest/workflows/statuses";

        sinon.assert.notCalled(this.triggerStub);

        this.sandbox.server.respondWith(new RegExp(URL), JSON.stringify(response));
        this.sandbox.server.respond();

        equal(this.sandbox.server.requests.length, 1, "An AJAX request was made to retrieve the statuses");
        equal(deferred.state(), "resolved", "The returned deferred was resolved");
        deferred.done(function (statuses) {
            deepEqual(statuses, response, "It was resolved with status information");
        });

        // The response should be cached.
        deferred = this.workflowStatusesAJAXManager.getStatuses({workflowName: 'workflow name'});

        sinon.assert.notCalled(this.triggerStub);

        equal(this.sandbox.server.requests.length, 1, "Subsequent calls don't result in an AJAX request");
        ok(this.sandbox.server.requests[0].url.indexOf('workflowName=workflow+name') !== -1);
        equal(deferred.state(), "resolved", "The returned deferred is resolved immediately");
        deferred.done(function (statuses) {
            deepEqual(statuses, response, "It is resolved with status information");
        });

        // Calling reset() should clear the cache.
        this.workflowStatusesAJAXManager.reset();
        deferred = this.workflowStatusesAJAXManager.getStatuses({workflowName: 'workflow name'});
        this.sandbox.server.respond();

        sinon.assert.notCalled(this.triggerStub);

        equal(this.sandbox.server.requests.length, 2, "reset() clears the cached response");
        equal(deferred.state(), "resolved", "The returned deferred was resolved");
    });

    test("getStatuses() rejects the returned deferred on error", function () {
        sinon.assert.notCalled(this.triggerStub);

        var deferred = this.workflowStatusesAJAXManager.getStatuses({workflowName: 'workflow name'}),
            URL = contextPath() + "/rest/workflowDesigner/latest/workflows/statuses";

        sinon.assert.notCalled(this.triggerStub);

        this.sandbox.server.respondWith(new RegExp(URL), [400, {}, "No deal."]);
        this.sandbox.server.respond();

        equal(deferred.state(), "rejected", "The returned deferred was rejected");
        deferred.fail(function (errorMessage) {
            equal(errorMessage, "No deal.", "It was rejected with the error message.");
        });
    });

    test("validateRemoveStatus() should not trigger 'sync' event", function () {
        sinon.assert.notCalled(this.triggerStub);

        this.workflowStatusesAJAXManager.validateRemoveStatus({
            statusId: "Open",
            workflowName: "Workflow"
        });

        sinon.assert.notCalled(this.triggerStub);
    });
});
