AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer-io",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var WorkflowAJAXManager = require("workflow-designer/io/ajax/workflow-ajax-manager");
    var WorkflowModel = require("workflow-designer/workflow-model");
    var contextPath = require("wrm/context-path");

    module("WorkflowAJAXManager", {
        setup: function () {
            this.sandbox = sinon.sandbox.create({useFakeServer: true});
        },

        teardown: function () {
            this.sandbox.restore();
        }
    });

    test("Successfully loading a workflow", function () {
        var deferred = WorkflowAJAXManager.load("Workflow"),
            expectedResponse;

        expectedResponse = {
            layout: simpleLayoutJSON,
            liveStepIds: [1, 2]
        };

        this.sandbox.server.respondWith(JSON.stringify(expectedResponse));
        this.sandbox.server.respond();

        equal(deferred.state(), "resolved", "The returned deferred is resolved on success");
        deferred.done(function (attributes, response) {
            equal(arguments.length, 2, "The deferred was resolved with two arguments");
            ok(attributes.updatedDate.getTime() === 100000, "The correct last updated date was loaded");
            deepEqual(attributes.updateAuthor, {
                userName: "bob",
                displayName: "Bob Dude"
            }, "The correct last updater details were loaded");
            deepEqual(attributes.liveStepIds, [1, 2], "The correct live step ID-s were loaded");
            equal(attributes.statuses.length, 6, "The correct number of statuses were loaded");
            equal(attributes.transitions.length, 4, "The correct number of transitions were loaded");
            deepEqual(response, expectedResponse, "The JSON response was passed to the done handler");
        });
    });

    test("Unsuccessfully loading a workflow because server rejected request.", function () {
        var deferred = WorkflowAJAXManager.load("workflow");
        this.sandbox.server.respond(); // 404

        equal(deferred.state(), "rejected", "The returned deferred is rejected on failure");
    });

    test("Successfully saving a workflow", function () {
        var deferred = WorkflowAJAXManager.save(new WorkflowModel());
        this.sandbox.server.respondWith(contextPath() + "/rest/workflowDesigner/latest/workflows", "");
        this.sandbox.server.respond();

        equal(deferred.state(), "resolved", "The returned deferred is resolved on success");
    });

    test("Unsuccessfully saving a workflow because server rejected request.", function () {
        var deferred = WorkflowAJAXManager.save(new WorkflowModel());
        this.sandbox.server.respondWith(contextPath() + "/rest/workflowDesigner/latest/workflows", [400, {}, "Page not found"]);
        this.sandbox.server.respond();

        equal(deferred.state(), "rejected", "The returned deferred is rejected on failure");
        deferred.fail(function (errorMessage) {
            equal(errorMessage, "Page not found", "Callback received the right response");
        });
    });

    test("Call to save a workflow was rejected because a draft was already being published or discarded.", function () {
        var savePromise,
            workflowModel = new WorkflowModel();

        WorkflowAJAXManager.publish(workflowModel);
        this.sandbox.server.respondWith(contextPath() + "/rest/workflowDesigner/latest/workflows/publishDraft", "");

        savePromise = WorkflowAJAXManager.save(workflowModel);
        this.sandbox.server.respond();

        equal(savePromise.state(), "pending", "The save promise is pending when a draft was already being published or discarded.");

        equal(this.sandbox.server.requests.length, 1, "Only the publish request was made");
        equal(this.sandbox.server.requests[0].url, contextPath() + "/rest/workflowDesigner/latest/workflows/publishDraft", "The only request is to publish");
    });

    test("A successful workflow publish returns a resolved promise", 1, function () {
        var publishPromise = WorkflowAJAXManager.publish(new WorkflowModel());
        this.sandbox.server.respondWith(contextPath() + "/rest/workflowDesigner/latest/workflows/publishDraft", "");
        this.sandbox.server.respond();

        equal(publishPromise.state(), "resolved", "The returned deferred is resolved on success");
    });

    test("Unsuccessfully publishing a workflow because server rejected request.", 2, function () {
        var promise = WorkflowAJAXManager.publish(new WorkflowModel());
        this.sandbox.server.respondWith(contextPath() + "/rest/workflowDesigner/latest/workflows/publishDraft", [400, {}, "Bad request"]);
        this.sandbox.server.respond();

        equal(promise.state(), "rejected", "The returned promise is rejected on failure");
        promise.fail(function (errorMessage) {
            equal(errorMessage, "Bad request", "Callback received the right response");
        });
    });

    test("A successful workflow discard returns a resolved promise", 1, function () {
        var publishPromise = WorkflowAJAXManager.discard("workflow");
        this.sandbox.server.respondWith(contextPath() + "/rest/workflowDesigner/latest/workflows/discardDraft", "");
        this.sandbox.server.respond();

        equal(publishPromise.state(), "resolved", "The returned deferred is resolved on success");
    });

    test("Unsuccessfully discarding a workflow because server rejected request.", 2, function () {
        var deferred = WorkflowAJAXManager.discard("Workflow");
        this.sandbox.server.respondWith(contextPath() + "/rest/workflowDesigner/latest/workflows/discardDraft", [400, {}, "Bad request"]);
        this.sandbox.server.respond();

        equal(deferred.state(), "rejected", "The returned deferred is rejected on failure");
        deferred.fail(function (errorMessage) {
            equal(errorMessage, "Bad request", "Callback received the right response");
        });
    });

});
