define("workflow-designer/io/ajax/statuses-ajax-manager", [
    "workflow-designer/io/ajax/ajax-manager",
    "workflow-designer/io/error-handler",
    "workflow-designer/underscore",
    "jquery",
    "wrm/context-path"
], function(
    AJAXManager,
    ErrorHandler,
    _,
    jQuery,
    contextPath
) {
    /**
     * Cache for statuses
     *
     * @inner
     * @type {Object|undefined}
     */
    var statuses;

    /**
     * Handles manipulating/retrieving statuses via AJAX.
     *
     * @namespace
     * @extends JIRA.WorkflowDesigner.IO.AJAX.AJAXManager
     */
    return new (AJAXManager.extend(/** @lends JIRA.WorkflowDesigner.IO.AJAX.StatusesAJAXManager */{
        /**
         * Retrieve information about all statuses in the system.
         *
         * The response is cached under the assumption that statuses don't change often.
         *
         * @return {jQuery.Deferred} A deferred that is resolved with status data.
         */
        getStatuses: function () {
            var deferred = jQuery.Deferred(),
                request;

            if (statuses) {
                deferred.resolve(statuses);
            } else {
                request = this._makeRequest({
                    dataType: "json",
                    url: contextPath() + "/rest/workflowDesigner/latest/statuses"
                });

                request.done(function (response) {
                    statuses = response;
                    deferred.resolve(statuses);
                });

                request.fail(function (response, status) {
                    deferred.reject(ErrorHandler.getErrorMessage(response, status));
                });
            }

            return deferred.promise();
        },

        /**
         * Retrieve the number of workflows a status is used in.
         *
         * @return {jQuery.Deferred} A deferred that is resolved with the number of associated workflows for the status represented
         * by the supplied statusId.
         */
        getNumberOfAssociatedWorkflows: function (statusId) {
            var deferred = jQuery.Deferred(),
                request;

            request = this._makeRequest({
                data: {"statusId": statusId},
                dataType: "json",
                url: contextPath() + "/rest/workflowDesigner/latest/statuses/associatedWorkflowCount"
            });

            request.done(function (response) {
                deferred.resolve(response);
            });

            request.fail(function (response, status) {
                deferred.reject(ErrorHandler.getErrorMessage(response, status));
            });

            return deferred.promise();
        },

        /**
         * Reset the manager, clearing its cache.
         */
        reset: function () {
            statuses = undefined;
        }
    }))();
});

AJS.namespace("JIRA.WorkflowDesigner.IO.AJAX.StatusesAJAXManager", null, require("workflow-designer/io/ajax/statuses-ajax-manager"));