define("workflow-designer/io/ajax/project-config-ajax-manager", [
    "workflow-designer/io/ajax/ajax-manager",
    "workflow-designer/io/error-handler",
    "workflow-designer/io/workflow-data-reader",
    "workflow-designer/backbone",
    "workflow-designer/underscore",
    "jquery",
    "wrm/context-path"
], function(
    AJAXManager,
    ErrorHandler,
    WorkflowDataReader,
    Backbone,
    _,
    jQuery,
    contextPath
) {
    var WORKFLOWS_REST_URL = contextPath() + "/rest/workflowDesigner/latest/workflows";

    /**
     * Processes the provided request. Parses response on success to layout data. On failure, it processes the response
     * and status into an error message
     *
     * @param {jQuery.jqXHR} request the request to handle
     * @returns {jQuery.Deferred} A promise that is resolved with the parsed workflow information and the JSON response.
     */
    function processResponse(request) {
        var data,
            deferred = jQuery.Deferred(),
            layout;

        request.done(function (response) {
            layout = response.layout;

            // The following lines for resolving the data are common between ProjectConfigAJAXManager and
            // WorkflowAJAXManager. They should be kept in sync.
            data = WorkflowDataReader.read(layout);
            data = _.extend(data, {
                liveStepIds: response.liveStepIds,
                permissions: response.workflowPermissions
            });

            deferred.resolve(response.isDraft, data, response);
        });

        request.fail(function (response, status) {
            deferred.reject(ErrorHandler.getErrorMessage(response, status));
        });
        return deferred.promise();
    }

    /**
     * An AJAX manager used to load workflow data in JIRA's project administration section.
     * If you change this, you'll likely need to change jira-project-config-plugin.
     *
     * @namespace
     * @extends JIRA.WorkflowDesigner.IO.AJAX.AJAXManager
     */
    return new (AJAXManager.extend(/** @lends JIRA.WorkflowDesigner.IO.AJAX.ProjectConfigAJAXManager */{
        /**
         * Loads the active workflow with the given name even if there is a draft.
         *
         * @param {string} options.name The name of the workflow to load.
         * @return {jQuery.Deferred} A promise that is resolved with the parsed workflow information and the JSON response.
         */
        loadWorkflow: function (options) {
            var data = _.extend({}, options);
            var request = this._makeRequest({
                data: data,
                dataType: "json",
                url: WORKFLOWS_REST_URL
            });

            return processResponse(request);
        },

        /**
         * Load the workflow with the given name. Loads the draft workflow data, if it is available; otherwise it loads
         * the active workflow data.
         *
         * @param {string} options.name The name of the workflow to load.
         * @return {jQuery.Deferred} A promise that is resolved with the parsed workflow information and the JSON response.
         */
        loadDraftWorkflowIfAvailable: function (options) {
            var data,
                request;

            data = _.extend({}, options, {
                preferDraft: true
            });

            request = this._makeRequest({
                data: data,
                dataType: "json",
                url: WORKFLOWS_REST_URL
            });

            return processResponse(request);
        },

        /**
         * Load the workflow with the given name. Loads the draft workflow data, and if it does not exist, it creates it;
         *
         * @param {string} options.name The name of the workflow to load.
         * @return {jQuery.Deferred} A promise that is resolved with the parsed workflow information and the JSON response.
         */
        loadOrCreateDraftWorkflow: function (options) {
            var data,
                request;

            data = _.extend({}, options, {
                draft: true
            });

            request = this._makeRequest({
                data: data,
                dataType: "json",
                url: WORKFLOWS_REST_URL
            });

            return processResponse(request);
        }
    }))();
});

AJS.namespace("JIRA.WorkflowDesigner.IO.AJAX.ProjectConfigAJAXManager", null, require("workflow-designer/io/ajax/project-config-ajax-manager"));