define("workflow-designer/io/workflow-data-writer", [
    "workflow-designer/underscore"
], function(
    _
) {

    /**
     * @inner
     * @param {JIRA.WorkflowDesigner.StatusModel} statusModel
     * @returns {Object}
     */
    function getStatusJSON(statusModel) {
        return _.chain(statusModel.toJSON())
                .pick("id", "x", "y")
                .value();
    }

    /**
     * @inner
     * @param {JIRA.WorkflowDesigner.TransitionModel} transitionModel
     * @returns {Object}
     */
    function getTransitionJSON(transitionModel) {
        var extension = {},
            isGlobalTransition,
            source,
            target;

        if (!transitionModel.isLoopedTransition()) {
            isGlobalTransition = transitionModel.isGlobalTransition();
            source = transitionModel.get("source");
            target = transitionModel.get("target");

            extension = {
                sourceId: isGlobalTransition ? target.id : source.id,
                targetId: target.id
            };
        }

        return _.chain(transitionModel.toJSON())
            .pick("id", "sourceAngle", "targetAngle")
            .extend(extension)
            .value();
    }

    /**
     * Writes workflow data to JSON.
     *
     * @namespace
     */
    return {
        /**
         * Write workflow data to JSON.
         *
         * @param {JIRA.WorkflowDesigner.WorkflowModel} workflowModel The workflow.
         * @return {object} The result of serializing <tt>workflowModel</tt> to JSON.
         */
        write: function (workflowModel) {
            var extension,
                layout,
                loopedTransitionContainer = workflowModel.get("loopedTransitionContainer");

            layout = {
                statuses: workflowModel.get("statuses").map(getStatusJSON),
                transitions: workflowModel.get("transitions").map(getTransitionJSON)
            };
            loopedTransitionContainer && _.extend(layout, {
                loopedTransitionContainer: {
                    x: loopedTransitionContainer.get("x"),
                    y: loopedTransitionContainer.get("y")
                }
            });

            extension = { layout: layout };

            return _.chain(workflowModel.toJSON())
                    .pick("draft", "name")
                    .extend(extension)
                    .value();
        }
    };
});

AJS.namespace("JIRA.WorkflowDesigner.IO.WorkflowDataWriter", null, require("workflow-designer/io/workflow-data-writer"));