define("workflow-designer/io/workflow-data-reader", [
    "workflow-designer/status-model",
    "workflow-designer/looped-transition-container-model",
    "workflow-designer/transition-model",
    "workflow-designer/underscore"
], function(
    StatusModel,
    LoopedTransitionContainerModel,
    TransitionModel,
    _
) {

    /**
     * @inner
     *
     * @param {Object} attributes
     * @param {Object} statusModels
     * @returns {JIRA.WorkflowDesigner.TransitionModel}
     */
    function createTransitionModel(attributes, statusModels) {
        attributes = _.chain(attributes).omit("sourceId", "targetId").extend({
            source: statusModels[attributes.sourceId],
            target: statusModels[attributes.targetId]
        }).value();

        return new TransitionModel(attributes);
    }

    /**
     * Loads workflow data.
     *
     * @namespace
     */
    return {
        /**
         * Read workflow information from JSON.
         *
         * @param {Object} data The JSON workflow data.
         * @return {object|undefined} The result of parsing <tt>data</tt> or <tt>undefined</tt> on failure.
         */
        read: function (data) {
            var loopedTransitionContainerModel,
                statusModels = {},
                transitionModels = [];

            if (!data || !data.statuses || !data.transitions) {
                return;
            }

            _.each(data.statuses, function (status) {
                var statusModel = new StatusModel(status);
                statusModels[statusModel.id] = statusModel;
            });

            _.each(data.transitions, function (transition) {
                transitionModels.push(createTransitionModel(transition, statusModels));
            });

            data.loopedTransitionContainer &&
                (loopedTransitionContainerModel = new LoopedTransitionContainerModel(data.loopedTransitionContainer));

            return {
                loopedTransitionContainer: loopedTransitionContainerModel,
                statuses: _.values(statusModels),
                transitions: transitionModels,
                updateAuthor: data.updateAuthor,
                updatedDate: new Date(data.updatedDate)
            };
        }
    };
});

AJS.namespace("JIRA.WorkflowDesigner.IO.WorkflowDataReader", null, require("workflow-designer/io/workflow-data-reader"));