define("workflow-designer/io/error-handler", [
    "workflow-designer/messages",
    "workflow-designer/underscore"
], function(
    Messages,
    _
) {
    var defaultErrorMessages;

    /**
     * @param {string} [responseText] The text of a response.
     * @param {number} [statusCode] The HTTP status code of a response.
     * @returns {boolean} Whether the response is a WebSudo error.
     */
    function isWebSudoError(responseText, statusCode) {
        return !!/websudo/i.exec(responseText) && statusCode === 401;
    }

    defaultErrorMessages = {
        generic: AJS.I18n.getText("common.forms.ajax.servererror"),
        notAuthenticated: AJS.I18n.getText("common.forms.ajax.unauthorised.alert"),
        timeout: AJS.I18n.getText("common.forms.ajax.timeout"),
        webSudo: AJS.I18n.getText("workflow.designer.websudo.timeout")
    };

    /**
     * Handles AJAX request errors.
     *
     * @namespace
     */
    var ErrorHandler = {
        /**
         * Given an AJAX response, returns an appropriate error message.
         *
         * @param {object} [response] The AJAX response.
         * @param {string} [response.responseText] Response text.
         * @param {number} [response.status] The HTTP response code.
         * @param {string} [status] A string describing the type of error that occurred.
         * @param {object} [errorMessages] Error message overrides.
         * @returns {string} An appropriate error message.
         */
        getErrorMessage: function (response, status, errorMessages) {
            var responseText = response && response.responseText,
                statusCode = response && response.status;

            errorMessages = _.defaults({}, errorMessages, defaultErrorMessages);

            if (status === "timeout") {
                return errorMessages.timeout;
            }

            if (isWebSudoError(responseText, statusCode)) {
                return errorMessages.webSudo;
            }

            if (statusCode === 401) {
                return errorMessages.notAuthenticated;
            }

            if (_.contains([400, 403, 404], statusCode)) {
                try {
                    // Handle serialised ErrorCollections.
                    return JSON.parse(responseText).errorMessages[0] || responseText;
                } catch (e) {
                    return responseText;
                }
            }

            if (status !== "abort") {
                return errorMessages.generic;
            }

            return null;
        },

        /**
         * Handle an AJAX error, possibly showing an error message.
         *
         * @param {object} [response] An AJAX response.
         * @param {string} [response.responseText] The text of the response.
         * @param {number} [response.status] The HTTP response code of the response.
         * @param {string} [status] A string describing the type of error that occurred.
         */
        handle: function (response, status) {
            var message;

            message = this.getErrorMessage(response, status);
            message && Messages.showErrorMessage(message);
        }
    };

    // Bind its methods so we can pass them around.
    _.bindAll(ErrorHandler, "getErrorMessage", "handle");

    return ErrorHandler;
});

AJS.namespace("JIRA.WorkflowDesigner.IO.ErrorHandler", null, require("workflow-designer/io/error-handler"));