/**
 * A Service that allows restricted access to instances of the JIRA.WorkflowDesigner.Application.
 *
 * When an instance of the Application is created, it is registered with this service. Consumers of this service
 * register a ready callback. The ready callback gets called with an API that provides limited access to the
 * instance of JIRA.WorkflowDesigner.Application.
 */
define("workflow-designer/internal-api/workflow-service", [
    "jquery",
    "exports"
], function (
    $,
    exports
) {

    var workflowDesigners = [];
    var readyCallbacks = [];

    /**
     * We do not want to export the entire workflow designer for public consumption. So this function is to restrict
     * the surface area of the api.
     *
     * @param {JIRA.WorkflowDesigner.Application} workflowDesigner
     */
    function getExports(workflowDesigner) {
        return {
            getWorkflowData: function () {
                return workflowDesigner.getWorkflowData();
            }
        };
    }

    /**
     * Iterates though all the ready callbacks, calling them for each registered workflow designer. It stores on
     * the callback the workflow designer instances it has executed for so, to avoid the ready callback being called
     * more than once per instance.
     */
    function executeCallbacks() {
        readyCallbacks.forEach(function (callback) {
            workflowDesigners.forEach(function (workflowDesigner) {
                callback.executedFor = callback.executedFor || [];
                if (callback.executedFor.indexOf(workflowDesigner) === -1) {
                    callback(getExports(workflowDesigner));
                    callback.executedFor.push(workflowDesigner);
                }
            });
        });
    }

    /**
     * Sets the workflow designer this service is to use.
     *
     * @param {JIRA.WorkflowDesigner.Application} workflowDesigner
     */
    exports.registerWorkflowDesigner = function (workflowDesigner) {
        workflowDesigners.push(workflowDesigner);
        executeCallbacks();
    };

    /**
     * Allow to pass in a callback to be executed when the service is ready a.k.a workflow designer has been set
     *
     * @param callback
     * @returns a promise
     */
    exports.ready = function (callback) {
        readyCallbacks.push(callback);
        executeCallbacks();
    };
});

AJS.namespace("JIRA.WorkflowDesigner.InternalApi.WorkflowService", null, require("workflow-designer/internal-api/workflow-service"));