/**
 * Triggers `state:*` events to reflect the current state of changes made in workflow designer.
 * There're three events:
 * - state:pending # triggered immediately after a request was made
 * - state:success # triggered after all requests completed and the last one succeed.
 * - state:failure # triggered when any of the requests failed.
 *
 * The last two have additional parameter "lastSyncTime" which is the time of the last successful save.
 */
define("workflow-designer/internal-api/save-status-service", [
    "workflow-designer/io/ajax/workflow-ajax-manager",
    "workflow-designer/io/ajax/workflow-global-transitions-ajax-manager",
    "workflow-designer/io/ajax/workflow-looped-transitions-ajax-manager",
    "workflow-designer/io/ajax/workflow-statuses-ajax-manager",
    "workflow-designer/io/ajax/workflow-transitions-ajax-manager",
    "workflow-designer/backbone",
    "workflow-designer/underscore"
], function(
    WorkflowAJAXManager,
    WorkflowGlobalTransitionsAJAXManager,
    WorkflowLoopedTransitionsAJAXManager,
    WorkflowStatusesAJAXManager,
    WorkflowTransitionsAJAXManager,
    Backbone,
    _
) {
    return Backbone.Marionette.Controller.extend({
        initialize: function() {
            this.counter = 0;
            this.lastSyncTime = 0;

            _.bindAll(this, "_successCallback", "_failureCallback");

            this.listenTo(WorkflowAJAXManager, "sync", this._syncEventHandler);
            this.listenTo(WorkflowGlobalTransitionsAJAXManager, "sync", this._syncEventHandler);
            this.listenTo(WorkflowLoopedTransitionsAJAXManager, "sync", this._syncEventHandler);
            this.listenTo(WorkflowStatusesAJAXManager, "sync", this._syncEventHandler);
            this.listenTo(WorkflowTransitionsAJAXManager, "sync", this._syncEventHandler);
        },

        _syncEventHandler: function(promise) {
            this.trigger("state:pending");
            ++this.counter;

            promise.then(this._successCallback);
            promise.fail(this._failureCallback);
        },

        _successCallback: function() {
            this.lastSyncTime = new Date().getTime();
            --this.counter;

            if (this.counter === 0) {
                this.trigger("state:success", this.lastSyncTime);
            }
        },

        _failureCallback: function() {
            --this.counter;
            this.trigger("state:failure", this.lastSyncTime);
        }
    });
});
