AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var FullScreenController = require("workflow-designer/full-screen-controller");
    var ProgressIndicatorBlanketView = require("workflow-designer/progress-indicator-blanket-view");
    var ActionsView = require("workflow-designer/actions-view");
    var ZoomControlView = require("workflow-designer/zoom-control-view");
    var PropertiesPanelView = require("workflow-designer/properties-panel/properties-panel-view");
    var CanvasView = require("workflow-designer/canvas-view");
    var TestUtilities = require("workflow-designer/test-utilities");
    var Canvas = TestUtilities.getCanvasClass();
    var jQuery = require("jquery");

    module("Canvas", {
        setup: function () {
            this.sandbox = sinon.sandbox.create();

            this.sandbox.spy(ProgressIndicatorBlanketView.prototype, "initialize");
            this.sandbox.spy(ActionsView.prototype, "initialize");
            this.sandbox.spy(ActionsView.prototype, "close");
            this.sandbox.spy(ZoomControlView.prototype, "initialize");
            this.sandbox.spy(PropertiesPanelView.prototype, "initialize");
            this.sandbox.spy(CanvasView.prototype, "initialize");

            this.sandbox.spy(Canvas.prototype, "initialize");
            this.sandbox.spy(Canvas.prototype, "hideProgressIndicator");
            this.sandbox.spy(Canvas.prototype, "showProgressIndicator");
        },

        spyModule: function (moduleKey) {
            this.sandbox.spy(require(moduleKey).prototype, "initialize");
        },

        teardown: function () {
            this.sandbox.restore();
        }
    });

    test("autoFit()", function () {
        var autoFitSpy = this.sandbox.spy(CanvasView.prototype, "autoFit"),
            canvas = TestUtilities.testCanvas();

        canvas.autoFit();
        equal(autoFitSpy.callCount, 1, "CanvasView#autoFit was called");
    });

    test("Creates an ActionsView when necessary", function () {
        TestUtilities.testCanvas();
        equal(ActionsView.prototype.initialize.callCount, 1, "An ActionsView was created");

        TestUtilities.testCanvas({actions: false});
        equal(ActionsView.prototype.initialize.callCount, 1, "No ActionsView was created");
    });

    test("Should initialize ActionView with proper default options", function () {
        var canvas = TestUtilities.testCanvas();
        var options = canvas.initialize.getCall(0).args[0];

        sinon.assert.calledWith(ActionsView.prototype.initialize, {
            canvasModel: canvas._canvasModel,
            fullScreenButton: true,
            immutable: false,
            workflowModel: options.workflowModel,
            saveStatusService: options.saveStatusService
        });
    });

    test("Should initialize ActionView with proper non-default options", function () {
        var canvas = TestUtilities.testCanvas({
            fullScreenButton: false,
            immutable: true
        });
        var options = canvas.initialize.getCall(0).args[0];

        sinon.assert.calledWith(ActionsView.prototype.initialize, {
            canvasModel: canvas._canvasModel,
            fullScreenButton: false,
            immutable: true,
            workflowModel: options.workflowModel,
            saveStatusService: options.saveStatusService
        });
    });

    test("Should render ActionViews", function () {
        var canvas = TestUtilities.testCanvas();

        var actionViewsEl = canvas.$el.find('.workflow-designer-section .actions');
        notEqual(actionViewsEl.html(), "", "rendered ActionViews content should not be empty");
    });

    test("Should close ActionViews properly", function () {
        var canvas = TestUtilities.testCanvas();

        sinon.assert.notCalled(canvas._actionsView.close);

        canvas.close();

        sinon.assert.calledOnce(canvas._actionsView.close);
    });

    test("Creates a PropertiesPanelView when necessary", function () {
        TestUtilities.testCanvas();
        equal(PropertiesPanelView.prototype.initialize.callCount, 1,
            "A PropertiesPanelView was created");

        TestUtilities.testCanvas({immutable: true});
        equal(PropertiesPanelView.prototype.initialize.callCount, 1,
            "No PropertiesPanelView was created");
    });

    test("Creates a ZoomControlView", function () {
        var element = jQuery("<div>").appendTo("#qunit-fixture"),
            renderSpy = this.sandbox.spy(ZoomControlView.prototype, "render"),
            zoomControlView;

        TestUtilities.testCanvas({el: element});
        zoomControlView = ZoomControlView.prototype.initialize.thisValues[0];

        ok(zoomControlView, "A ZoomControlView was created");
        equal(renderSpy.callCount, 1, "Its render method was called");
        ok(jQuery.contains(element[0], zoomControlView.el), "Its element was appended to the workflow designer container");
    });

    test("Entering and exiting full screen", function () {
        var actionsView,
            container = jQuery("<div>").appendTo("#qunit-fixture"),
            enterFullScreenDeferred = jQuery.Deferred(),
            enterFullScreenStub = this.sandbox.stub(FullScreenController, "enterFullScreen").returns(enterFullScreenDeferred.promise()),
            exitFullScreenDeferred = jQuery.Deferred(),
            exitFullScreenStub = this.sandbox.stub(FullScreenController, "exitFullScreen").returns(exitFullScreenDeferred.promise());

        TestUtilities.testCanvas({el: container});
        actionsView = ActionsView.prototype.initialize.thisValues[0];

        actionsView.trigger("fullScreen:enter");
        equal(enterFullScreenStub.callCount, 1, "FullScreenController.enterFullScreen() was called");
        ok(enterFullScreenStub.args[0][0] === container[0], "It was passed the correct arguments");

        enterFullScreenDeferred.resolve();

        actionsView.trigger("fullScreen:exit");
        equal(exitFullScreenStub.callCount, 1, "FullScreenController.exitFullScreen() was called");
        ok(exitFullScreenStub.args[0][0] === container[0], "It was passed the correct arguments");

        exitFullScreenDeferred.resolve();
    });

    test("Hiding and showing a progress indicator", function () {
        var canvas = TestUtilities.testCanvas(),
            progressIndicatorBlanketViewSpy = ProgressIndicatorBlanketView.prototype.initialize,
            hideSpy = this.sandbox.spy(ProgressIndicatorBlanketView.prototype, "hide"),
            setLockedSpy = this.sandbox.spy(CanvasView.prototype, "setLocked"),
            showSpy = this.sandbox.spy(ProgressIndicatorBlanketView.prototype, "show");

        canvas.showProgressIndicator();
        ok(setLockedSpy.calledWith(true), "The CanvasView was locked");
        equal(progressIndicatorBlanketViewSpy.callCount, 1, "A ProgressIndicatorBlanketView was created");
        ok(progressIndicatorBlanketViewSpy.args[0][0].container === canvas.ui.container, "It used the canvas container");
        equal(showSpy.callCount, 1, "The ProgressIndicatorBlanketView was shown");

        canvas.hideProgressIndicator();
        ok(setLockedSpy.calledWith(false), "The CanvasView was unlocked");
        equal(hideSpy.callCount, 1, "The ProgressIndicatorBlanketView was hidden");
    });

    test("Shows a progress indicator while adding a status", function () {
        var actionsView, canvas;

        canvas = TestUtilities.testCanvas();
        actionsView = ActionsView.prototype.initialize.thisValues[0];

        actionsView.trigger("addStatus:submit");
        equal(canvas.showProgressIndicator.callCount, 1, "Canvas#showProgressIndicator was called");

        actionsView.trigger("addStatus:done");
        equal(canvas.hideProgressIndicator.callCount, 1, "Canvas#hideProgressIndicator was called");
    });

    test("Shows a progress indicator while the properties panel is processing a transaction", function () {
        var canvas, propertiesPanelView;

        canvas = TestUtilities.testCanvas();
        propertiesPanelView = PropertiesPanelView.prototype.initialize.thisValues[0];

        propertiesPanelView.trigger("transaction:start");
        equal(canvas.showProgressIndicator.callCount, 1, "Canvas#showProgressIndicator was called");

        propertiesPanelView.trigger("transaction:end");
        equal(canvas.hideProgressIndicator.callCount, 1, "Canvas#hideProgressIndicator was called");
    });

    test("Zooms when the ZoomControlView fires zoomIn/Out events", function () {
        var zoomControlView,
            zoomSpy = this.sandbox.stub(CanvasView.prototype, "zoom");

        TestUtilities.testCanvas();
        zoomControlView = ZoomControlView.prototype.initialize.thisValues[0];

        zoomControlView.trigger("zoomIn");
        equal(zoomSpy.callCount, 1, "CanvasView#zoom was called");
        deepEqual(zoomSpy.args[0], [0.75], "It was passed the correct arguments");

        zoomControlView.trigger("zoomOut");
        equal(zoomSpy.callCount, 2, "CanvasView#zoom was called");
        deepEqual(zoomSpy.args[1], [1.25], "It was passed the correct arguments");
    });

});
