define("workflow-designer/application", [
    "workflow-designer/io/layout-auto-saver",
    "workflow-designer/workflow-model",
    "workflow-designer/browser-support",
    "workflow-designer/messages",
    "workflow-designer/io/ajax/workflow-ajax-manager",
    "workflow-designer/canvas",
    "workflow-designer/templates",
    "workflow-designer/backbone",
    "workflow-designer/underscore",
    "workflow-designer/internal-api/workflow-service",
    "workflow-designer/internal-api/save-status-service",
    "jquery"
], function(
    LayoutAutoSaver,
    WorkflowModel,
    BrowserSupport,
    Messages,
    WorkflowAJAXManager,
    Canvas,
    Templates,
    Backbone,
    _,
    WorkflowService,
    SaveStatusService,
    jQuery
) {
    /**
     * Show a "browser not supported" message in an element.
     *
     * @param {element} element The element to show the message in.
     */
    function showBrowserNotSupportedWarning(element) {
        var template = Templates.browserNotSupportedWarning;
        jQuery(template()).appendTo(element);
    }

    /**
     * @class
     * @classdesc The JIRA workflow designer.
     * @constructor
     * @param {object} options
     * @param {boolean} [options.actions=true] Whether to show the actions bar.
     * @param {number} [options.currentStepId] The step ID of the status that is to be highlighted as "current".
     * @param {boolean} [options.draft=false] Whether the draft version of the workflow described by <tt>workflowId</tt> should be loaded.
     * @param {element} options.element The element to show the workflow designer in.
     * @param {boolean} [options.fullScreenButton=true] Whether full screen mode is enabled.
     * @param {boolean} [options.immutable=false] Whether the workflow designer should be read-only.
     * @param {object} [options.layoutData] The layout data to display.
     * @param {string} [options.workflowId] The ID of the workflow to load.
     * @param {boolean} [options.isDraftWithChanges=false] don't show last-saved-by for a draft without changes
     */
    var Application = function (options) {
        options = _.defaults({}, options, {
            draft: false,
            immutable: false
        });

        if (!BrowserSupport.browserIsSupported()) {
            showBrowserNotSupportedWarning(options.element);
            return;
        }

        this._workflowModel = new WorkflowModel({
            currentStepId: options.currentStepId,
            draft: options.draft,
            name: options.workflowId,
            analyticsData: options.analyticsData
        });
        this._isDraftWithChanges = options.isDraftWithChanges;

        this._saveStatusService = new SaveStatusService();
        this.listenTo(this._saveStatusService, "state:success", function() {
            this.trigger("sync");
        });

        this._canvas = this._createCanvas(options).render();

        if (!options.immutable) {
            this._layoutAutoSaver = new LayoutAutoSaver({
                workflowModel: this._workflowModel
            });
        }

        this._initialiseWorkflow(options);
    };

    _.extend(Application.prototype, Backbone.Events,
    /** @lends JIRA.WorkflowDesigner.Application# */
    {
        /**
         * @param {object} options Options to pass to the constructor.
         * @private
         * @return {JIRA.WorkflowDesigner.Canvas} A `JIRA.WorkflowDesigner.Canvas`.
         */
        _createCanvas: function (options) {
            options = _.extend(options, {
                el: options.element,
                workflowModel: this._workflowModel,
                saveStatusService: this._saveStatusService
            });

            return new Canvas(options);
        },

        /**
         * Destroy the workflow designer.
         */
        destroy: function () {
            this.stopListening();
            this._canvas && this._canvas.close();
            this._layoutAutoSaver && this._layoutAutoSaver.destroy();
            this._saveStatusService && this._saveStatusService.close();
        },

        /**
         * Discard the draft of the workflow that is currently shown in the workflow designer.
         * When called, this function will disable the workflow designer until the operation has completed.
         * If a failure occurred, the workflow designer will be enabled again and a message will be shown to the user.
         *
         * @param {object} [options]
         * @param {boolean} [options.reloadDesigner=true] Whether the workflow designer should be reloaded with a new draft after the operation.
         * @returns {jQuery.Promise} A promise that is resolved on success.
         */
        discardDraft: function(options) {
            options = _.defaults({}, options, {
                reloadDesigner: true
            });
            this._canvas.showProgressIndicator();

            return WorkflowAJAXManager.discard(this._workflowModel.get("name"))
                    .pipe(_.bind(this._onPublishOrDiscardSuccessHandler, this, options.reloadDesigner),
                            _.bind(this._onPublishOrDiscardFailHandler, this));
        },

        /**
         * Publish the draft of the workflow that is currently shown in the workflow designer.
         * When called, this function will disable the workflow designer until the operation has completed.
         * If a failure occurred, the workflow designer will be enabled again and a message will be shown to the user.
         *
         * @param {object} [options]
         * @param {boolean} [options.reloadDesigner=true] Whether the workflow designer should be reloaded with a new draft after the operation.
         * @returns {jQuery.Promise} A promise that is resolved on success.
         */
        publishDraft: function(options) {
            options = _.defaults({}, options, {
                reloadDesigner: true
            });
            this._canvas.showProgressIndicator();

            return WorkflowAJAXManager.publish(this._workflowModel)
                    .pipe(_.bind(this._onPublishOrDiscardSuccessHandler, this, options.reloadDesigner),
                            _.bind(this._onPublishOrDiscardFailHandler, this));
        },

        /**
         * Initialise a workflow into the designer.
         * @param {object} options
         * @param {boolean} [options.draft=false] Whether the draft version of the workflow described by <tt>workflowId</tt> should be loaded.
         * @param {object} [options.layoutData] The workflow data to load.
         * @param {string} [options.workflowId] The ID of the workflow to load.
         * @private
         */
        _initialiseWorkflow: function (options) {
            var instance = this,
                request = this._loadWorkflow(options);

            request.done(function () {
                instance._workflowModel.set("loadedAt", new Date());
                instance._canvas.autoFit();
                WorkflowService.registerWorkflowDesigner(instance);
                instance.trigger("loaded");
            });
        },

        /**
         * Load a workflow into the designer.
         *
         * @param {object} options
         * @param {boolean} [options.draft=false] Whether the draft version of the workflow described by <tt>workflowId</tt> should be loaded.
         * @param {object} [options.layoutData] The workflow data to load.
         * @param {string} [options.workflowId] The ID of the workflow to load.
         * @returns {jQuery.Promise} A promise that is resolved on success.
         * @private
         */
        _loadWorkflow: function (options) {
            var deferred = jQuery.Deferred(),
                instance = this,
                projectKey = jQuery("meta[name=projectKey]").attr("content"),
                request;

            if (options.layoutData) {
                request = jQuery.Deferred().resolve(options.layoutData);
            } else if (options.workflowId) {
                request = WorkflowAJAXManager.load(options.workflowId, !!options.draft);
            }

            if (request) {
                this._canvas.showProgressIndicator();

                request.always(function () {
                    WorkflowAJAXManager.triggerMauEventForProject(projectKey);
                    instance._canvas.hideProgressIndicator();
                });

                request.done(function (layoutData) {
                    if (!instance._isDraftWithChanges) {
                        layoutData.updatedDate = null;
                    }
                    instance._workflowModel.reset(layoutData);
                    deferred.resolve();
                });

                request.fail(function (errorMessage) {
                    Messages.showErrorMessage(errorMessage);
                    deferred.reject(errorMessage);
                });
            } else {
                deferred.resolve();
            }

            return deferred.promise();
        },

        _onPublishOrDiscardSuccessHandler: function (reloadDesigner) {
            this._layoutAutoSaver.removeUnsavedChangesMessage();
            if (reloadDesigner) {
                return this._loadWorkflow({
                    workflowId: this._workflowModel.get("name"),
                    draft: this._workflowModel.get("draft")
                });
            }
        },

        _onPublishOrDiscardFailHandler: function (errorMessage) {
            Messages.showErrorMessage(errorMessage);
            this._canvas.hideProgressIndicator();
        },
        /**
         * Gives back a javascript object tree representing the current state of the workflow designer.
         * E.g the workflow name transitions and statuses.
         *
         * @returns {object} object tree
         */
        getWorkflowData: function () {
            return this._workflowModel.toJSON();
        }
    });

    return Application;
});

AJS.namespace("JIRA.WorkflowDesigner.Application", null, require("workflow-designer/application"));
