AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer"
], function () {

    var Application;
    var Canvas;
    var WorkflowModel;
    var WorkflowAJAXManager;
    var Messages;
    var LayoutAutoSaver;
    var BrowserSupport;
    var Templates;
    var jQuery = require("jquery");
    
    module("Application", {
        setup: function () {
            this.sandbox = sinon.sandbox.create();
            this.context = AJS.test.mockableModuleContext();

            var workflowAJAXManager = require("workflow-designer/io/ajax/workflow-ajax-manager");
            this.loadWorkflowStub = this.sandbox.stub(workflowAJAXManager, "load");
            this.publishStub = this.sandbox.stub(workflowAJAXManager, "publish");
            this.discardStub = this.sandbox.stub(workflowAJAXManager, "discard");
            this.mauStub = this.sandbox.stub(workflowAJAXManager, "triggerMauEventForProject");
            this.context.mock("workflow-designer/io/ajax/workflow-ajax-manager", workflowAJAXManager);
            WorkflowAJAXManager = this.context.require("workflow-designer/io/ajax/workflow-ajax-manager");

            this.context.mock("workflow-designer/browser-support", require("workflow-designer/browser-support"));
            BrowserSupport = this.context.require("workflow-designer/browser-support");

            this.context.mock("workflow-designer/io/layout-auto-saver", require("workflow-designer/io/layout-auto-saver"));
            LayoutAutoSaver = this.context.require("workflow-designer/io/layout-auto-saver");

            this.context.mock("workflow-designer/templates", require("workflow-designer/templates"));
            Templates = this.context.require("workflow-designer/templates");

            this.context.mock("workflow-designer/messages", require("workflow-designer/messages"));
            Messages = this.context.require("workflow-designer/messages");

            var WorkflowModelClass = require("workflow-designer/workflow-model");
            this.workflowModelResetSpy = this.sandbox.spy(WorkflowModelClass.prototype, "reset");
            var workflowModel = new WorkflowModelClass();
            this.context.mock("workflow-designer/workflow-model", sinon.stub().returns(workflowModel));
            WorkflowModel = this.context.require("workflow-designer/workflow-model");

            Canvas = require("workflow-designer/canvas");
            this.sandbox.spy(Canvas.prototype, "initialize");
            this.sandbox.spy(Canvas.prototype, "hideProgressIndicator");
            this.sandbox.spy(Canvas.prototype, "showProgressIndicator");
            this.sandbox.spy(Canvas.prototype, "autoFit");
            this.context.mock("workflow-designer/canvas", Canvas);
        },
        teardown: function () {
            this.sandbox.restore();
        },
        prepareApplication: function() {
            Application = this.context.require("workflow-designer/application");
        },
        getFixtureEl: function () {
            return jQuery("<div/>").appendTo("#qunit-fixture").css("height", 500);
        }
    });

    test("A global message is shown on error", function () {
        var errorMessage = "No!",
            showErrorMessageSpy = this.sandbox.stub(Messages, "showErrorMessage");
        this.prepareApplication();

        this.loadWorkflowStub.returns(jQuery.Deferred().reject(errorMessage));
        new Application({workflowId: "Workflow", element: this.getFixtureEl()});

        equal(showErrorMessageSpy.callCount, 1, "Messages.showErrorMessage was called");
        deepEqual(showErrorMessageSpy.args[0], [errorMessage], "It was passed the correct arguments");
    });

    test("A warning is shown in unsupported browsers", function () {
        var element = this.getFixtureEl(),
            supportedStub = this.sandbox.stub(BrowserSupport, "browserIsSupported").returns(false),
            templateSpy = this.sandbox.spy(Templates, "browserNotSupportedWarning");

        this.prepareApplication();
        new Application({element: element});

        sinon.assert.notCalled(Canvas.prototype.initialize);
        equal(supportedStub.callCount, 1, "BrowserSupport.browserIsSupported was called");
        equal(templateSpy.callCount, 1, "JIRA.WorkflowDesigner.Templates.browserNotSupportWarning was called");
        equal(element.find(".aui-message.warning").length, 1, "The warning was appended to the application's element");
    });

    test("destroy() calls LayoutAutoSaver#destroy", function () {
        this.prepareApplication();
        var destroySpy = this.sandbox.spy(LayoutAutoSaver.prototype, "destroy");

        new Application({element: this.getFixtureEl()}).destroy();
        equal(destroySpy.callCount, 1, "LayoutAutoSaver#destroy was called");
    });

    test("destroy() doesn't throw an exception in unsupported browsers", function () {
        this.prepareApplication();
        var application;

        expect(0);
        this.sandbox.stub(BrowserSupport, "browserIsSupported").returns(false);

        application = new Application({element: this.getFixtureEl()});
        application.destroy();
    });

    test("Passing layout data to the constructor", function () {
        this.prepareApplication();
        new Application({layoutData: {}, element: this.getFixtureEl()});
        equal(this.workflowModelResetSpy.callCount, 1, "The Application's WorkflowModel was reset");
    });

    test("Should clean-up SaveStatusService when being destroyed", function () {
        this.prepareApplication();
        var app = new Application({element: this.getFixtureEl()});
        var saveStatusService = app._saveStatusService;

        this.sandbox.spy(saveStatusService, "close");
        var syncTriggerSpy = this.sandbox.stub(app, "trigger").withArgs("sync");

        app.destroy();

        sinon.assert.calledOnce(saveStatusService.close);
        sinon.assert.notCalled(syncTriggerSpy);

        saveStatusService.trigger("state:success");

        sinon.assert.notCalled(syncTriggerSpy);
    });

    test("Should close canvas when being destroyed", function () {
        this.prepareApplication();
        var app = new Application({element: this.getFixtureEl()});
        var closeSpy = this.sandbox.spy(app._canvas, "close");

        sinon.assert.notCalled(closeSpy);

        app.destroy();

        sinon.assert.calledOnce(closeSpy);
    });

    test("Should pass SaveStatusService to Canvas", function () {
        this.prepareApplication();
        var app = new Application({element: this.getFixtureEl()});

        sinon.assert.calledOnce(app._canvas.initialize);

        var options = app._canvas.initialize.getCall(0).args[0];
        strictEqual(options.workflowModel, app._workflowModel, "incorrect WorkflowModel");
        strictEqual(options.saveStatusService, app._saveStatusService, "incorrect SaveStatusService");
    });

    test("Should listen on 'state:success' of SaveStatusService and trigger 'sync' event", function () {
        this.prepareApplication();
        var app = new Application({element: this.getFixtureEl()});
        var saveStatusService = app._saveStatusService;
        var syncTriggerSpy = this.sandbox.stub(app, "trigger").withArgs("sync");

        sinon.assert.notCalled(syncTriggerSpy);

        saveStatusService.trigger("state:success");
        sinon.assert.calledOnce(syncTriggerSpy);

        saveStatusService.trigger("state:success");
        sinon.assert.calledTwice(syncTriggerSpy);
    });

    test("Should set `updatedDate` to null when `isDraftWithChanges=false` was passed to the constructor", function () {
        this.prepareApplication();
        new Application({isDraftWithChanges: false, layoutData: {}, element: this.getFixtureEl()});

        sinon.assert.calledOnce(this.workflowModelResetSpy);
        deepEqual(this.workflowModelResetSpy.args[0], [{updatedDate: null}]);
    });

    test("Should set `updatedDate` to null when `isDraftWithChanges` was not passed to the constructor", function () {
        this.prepareApplication();
        new Application({layoutData: {}, element: this.getFixtureEl()});

        sinon.assert.calledOnce(this.workflowModelResetSpy);
        deepEqual(this.workflowModelResetSpy.args[0], [{updatedDate: null}]);
    });

    test("Should not change `updatedDate` when `isDraftWithChanges=true` was passed to the constructor", function () {
        this.prepareApplication();
        new Application({isDraftWithChanges: true, layoutData: {}, element: this.getFixtureEl()});

        sinon.assert.calledOnce(this.workflowModelResetSpy);
        deepEqual(this.workflowModelResetSpy.args[0], [{}]);
    });

    test("Supports loading a workflow by ID", function () {
        var deferred = jQuery.Deferred();
        var workflowModel = WorkflowModel();

        this.loadWorkflowStub.returns(deferred.promise());

        this.prepareApplication();
        var app = new Application({workflowId: "Workflow", element: this.getFixtureEl()});

        equal(this.loadWorkflowStub.callCount, 1, "WorkflowAJAXManager#load() was called");
        deepEqual(this.loadWorkflowStub.args[0], ["Workflow", false], "It was passed the correct arguments");
        equal(app._canvas.showProgressIndicator.callCount, 1, "Canvas#showProgressIndicator() was called");

        deferred.resolve({});
        equal(app._canvas.autoFit.callCount, 1, "Canvas#autoFit() was called");
        equal(app._canvas.hideProgressIndicator.callCount, 1, "Canvas#hideProgressIndicator() was called");
        equal(this.mauStub.callCount, 1, "WorkflowAJAXManager#triggerMauEventForProject() was called");
        equal(this.workflowModelResetSpy.callCount, 1, "WorkflowModel#reset() was called");
        deepEqual(this.workflowModelResetSpy.args[0], [{updatedDate: null}], "It was passed the correct arguments");
        ok(workflowModel.has("loadedAt"), "WorkflowModel's loadedAt property was set");
        ok(workflowModel.get("loadedAt") instanceof Date, "It is a Date object");
        ok(new Date() - workflowModel.get("loadedAt") < 500, "It is the current time");
    });

    test("The current step ID, if given, is passed to WorkflowModel", function () {
        this.prepareApplication();
        this.workflowModelSpy = WorkflowModel;
        new Application({currentStepId: 42, element: this.getFixtureEl()});

        equal(this.workflowModelSpy.args[0][0].currentStepId, 42, "The current step ID was passed to the WorkflowModel constructor");
    });

    test("Triggers a \"loaded\" event after successfully loading a workflow via AJAX", function () {
        this.prepareApplication();
        var application,
            deferred = jQuery.Deferred(),
            loadedSpy = sinon.spy();

        this.loadWorkflowStub.returns(deferred.promise());
        application = new Application({workflowId: "Workflow", element: this.getFixtureEl()});
        application.on("loaded", loadedSpy);

        deferred.resolve({});
        equal(loadedSpy.callCount, 1, "A \"loaded\" event was triggered");
    });

    test("Triggers a \"loaded\" event after successfully loading the given workflow data", function () {
        this.prepareApplication();
        var triggerSpy = this.sandbox.spy(Application.prototype, "trigger");

        new Application({layoutData: {}, element: this.getFixtureEl()});
        ok(triggerSpy.calledWithExactly("loaded"), "A \"loaded\" event was triggered");
    });

    test("publishDraft() successfully publishes draft", function () {
        this.prepareApplication();
        var app = new Application({workflowId: "Workflow", element: this.getFixtureEl()}),
            resultDeferred,
            deferred = jQuery.Deferred(),
            loadingDeferred = jQuery.Deferred().resolve(),
            showErrorMessageSpy = this.sandbox.stub(Messages, "showErrorMessage"),
            loadWorkflowStub = this.sandbox.stub(Application.prototype, "_loadWorkflow");

        this.publishStub.returns(deferred.promise());
        loadWorkflowStub.returns(loadingDeferred);

        resultDeferred = app.publishDraft();
        equal(app._canvas.showProgressIndicator.callCount, 1, "Canvas#showProgressIndicator() was called");
        equal(resultDeferred.state(), "pending", "The returned deferred is pending");
        equal(this.publishStub.callCount, 1, "WorkflowAJAXManager#publish() was called");

        deferred.resolve();
        equal(resultDeferred.state(), "resolved", "The returned deferred is resolved on success");
        equal(app._canvas.hideProgressIndicator.callCount, 0, "Canvas#hideProgressIndicator() was not called");
        equal(showErrorMessageSpy.callCount, 0, "Messages.showErrorMessage was not called");
        equal(loadWorkflowStub.callCount, 1, "Workflow designer was reloaded");
    });

    test("publishDraft() without reloadDesigner flag successfully publishes draft", function () {
        this.prepareApplication();
        var app = new Application({workflowId: "Workflow", element: this.getFixtureEl()}),
            resultDeferred,
            deferred = jQuery.Deferred(),
            showErrorMessageSpy = this.sandbox.stub(Messages, "showErrorMessage"),
            loadWorkflowStub = this.sandbox.stub(Application.prototype, "_loadWorkflow");

        this.publishStub.returns(deferred.promise());

        resultDeferred = app.publishDraft({reloadDesigner: false});
        equal(app._canvas.showProgressIndicator.callCount, 1, "Canvas#showProgressIndicator() was called");
        equal(resultDeferred.state(), "pending", "The returned deferred is pending");
        equal(this.publishStub.callCount, 1, "WorkflowAJAXManager#publish() was called");

        deferred.resolve();
        equal(resultDeferred.state(), "resolved", "The returned deferred is resolved on success");
        equal(app._canvas.hideProgressIndicator.callCount, 0, "Canvas#hideProgressIndicator() was not called");
        equal(showErrorMessageSpy.callCount, 0, "Messages.showErrorMessage was not called");
        equal(loadWorkflowStub.callCount, 0, "Workflow designer was not reloaded");
    });

    test("publishDraft() fails publishing draft", function () {
        this.prepareApplication();
        var app = new Application({workflowId: "Workflow", element: this.getFixtureEl()}),
            resultDeferred,
            deferred = jQuery.Deferred(),
            showErrorMessageSpy = this.sandbox.stub(Messages, "showErrorMessage");

        this.publishStub.returns(deferred.promise());

        resultDeferred = app.publishDraft();
        equal(resultDeferred.state(), "pending", "The returned deferred is pending");
        equal(this.publishStub.callCount, 1, "WorkflowAJAXManager#publish() was called");
        equal(app._canvas.showProgressIndicator.callCount, 1, "Canvas#showProgressIndicator() was called");

        deferred.reject("error message");
        equal(resultDeferred.state(), "rejected", "The returned deferred is rejected on failure");
        equal(app._canvas.hideProgressIndicator.callCount, 1, "Canvas#hideProgressIndicator() was called");
        equal(showErrorMessageSpy.callCount, 1, "Messages.showErrorMessage was called");
    });

    test("discardDraft() successfully discards draft", function () {
        this.prepareApplication();
        var app = new Application({workflowId: "Workflow", element: this.getFixtureEl()}),
            resultDeferred,
            deferred = jQuery.Deferred(),
            showErrorMessageSpy = this.sandbox.stub(Messages, "showErrorMessage"),
            loadWorkflowStub = this.sandbox.stub(Application.prototype, "_loadWorkflow");

        this.discardStub.returns(deferred.promise());
        loadWorkflowStub.returns(jQuery.Deferred().resolve());

        resultDeferred = app.discardDraft();
        equal(resultDeferred.state(), "pending", "The returned deferred is pending");
        equal(app._canvas.showProgressIndicator.callCount, 1, "Canvas#showProgressIndicator() was called");
        equal(this.discardStub.callCount, 1, "WorkflowAJAXManager#discard() was called");

        deferred.resolve();
        equal(resultDeferred.state(), "resolved", "The returned deferred is resolved on success");
        equal(app._canvas.hideProgressIndicator.callCount, 0, "Canvas#hideProgressIndicator() was not called");
        equal(showErrorMessageSpy.callCount, 0, "Messages.showErrorMessage was not called");
        equal(this.discardStub.callCount, 1, "WorkflowAJAXManager#discard() was called");
        equal(loadWorkflowStub.callCount, 1, "Workflow designer was reloaded");
    });

    test("discardDraft() without reloadDesigner flag successfully discards draft", function () {
        this.prepareApplication();
        var app = new Application({workflowId: "Workflow", element: this.getFixtureEl()}),
            resultDeferred,
            deferred = jQuery.Deferred(),
            showErrorMessageSpy = this.sandbox.stub(Messages, "showErrorMessage"),
            loadWorkflowStub = this.sandbox.stub(Application.prototype, "_loadWorkflow");

        this.discardStub.returns(deferred.promise());
        loadWorkflowStub.returns(jQuery.Deferred().resolve());

        resultDeferred = app.discardDraft({reloadDesigner: false});
        equal(resultDeferred.state(), "pending", "The returned deferred is pending");
        equal(app._canvas.showProgressIndicator.callCount, 1, "Canvas#showProgressIndicator() was called");
        equal(this.discardStub.callCount, 1, "WorkflowAJAXManager#discard() was called");

        deferred.resolve();
        equal(resultDeferred.state(), "resolved", "The returned deferred is resolved on success");
        equal(app._canvas.hideProgressIndicator.callCount, 0, "Canvas#hideProgressIndicator() was not called");
        equal(showErrorMessageSpy.callCount, 0, "Messages.showErrorMessage was not called");
        equal(loadWorkflowStub.callCount, 0, "Workflow designer was not reloaded");
    });

    test("discardDraft() fails discarding draft", function () {
        this.prepareApplication();
        var app = new Application({workflowId: "Workflow", element: this.getFixtureEl()}),
            resultDeferred,
            deferred = jQuery.Deferred(),
        showErrorMessageSpy = this.sandbox.stub(Messages, "showErrorMessage");

        this.discardStub.returns(deferred.promise());

        resultDeferred = app.discardDraft();
        equal(resultDeferred.state(), "pending", "The returned deferred is pending");
        equal(app._canvas.showProgressIndicator.callCount, 1, "Canvas#showProgressIndicator() was called");

        deferred.reject("error message");
        equal(resultDeferred.state(), "rejected", "The returned deferred is rejected on failure");
        equal(app._canvas.hideProgressIndicator.callCount, 1, "Canvas#hideProgressIndicator() was called");
        equal(showErrorMessageSpy.callCount, 1, "Messages.showErrorMessage was called");
    });
});
