/**
 * Utilities for getting, setting and checking metadata stored in the &lt;head&rt; of the document.
 *
 * @module jira/util/data/meta
 */
define('jira/util/data/meta', ['jquery', 'jira/util/strings', 'jira/util/data/meta/store'], function (jQuery, Strings, store) {
    'use strict';

    return {
        /**
         * Sets metadata with a key and value, for use when the state of the page changes after
         * loading from the server
         * @param key
         * @param value
         */
        set: function set(key, value) {
            store[key] = value;
        },


        /**
         * Returns a value given a key. If no entry exists with the key, undefined is returned.
         * If the string value is "true" or "false" the respective boolean value is returned.
         *
         * @param key
         * @return {String|boolean|undefined}
         */
        get: function get(key) {
            if (typeof store[key] !== "undefined") {
                return store[key];
            }

            var metaEl = jQuery('meta[name=\'ajs-' + key + '\']');
            if (!metaEl.length) {
                return undefined;
            }

            var value = metaEl.attr("content");
            return Strings.asBooleanOrString(value);
        },


        /**
         * Returns true if the value for the provided key is equal to "true", else returns false.
         *
         * @param key
         * @return {boolean}
         */
        getBoolean: function getBoolean(key) {
            return this.get(key) === true;
        },


        /**
         * Returns a number if the value for the provided key can be converted to one.
         * Good for retrieving content ids to check truthiness (e.g. '0' is truthy but 0 is falsy).
         *
         * @param key
         * @return {number}
         */
        getNumber: function getNumber(key) {
            return +this.get(key);
        },


        /**
         * Mainly for use when debugging, returns all Data pairs in a map for eyeballing.
         * @return {Object}
         */
        getAllAsMap: function getAllAsMap() {
            var map = {};
            jQuery("meta[name^=ajs-]").each(function () {
                map[this.name.substring(4)] = this.content;
            });
            return jQuery.extend(map, store);
        },
        getLanguage: function getLanguage() {
            var locale = this.get("user-locale");
            var language;
            if (Strings.contains(locale, "_")) {
                language = locale.split("_")[0];
            }
            return language || "en";
        }
    };
});

define('jira/util/data/meta/store', {});