define('jira/newsletter/signuptip', ['jira/analytics', 'jira/util/formatter', 'jira/util/data/meta', 'jira/newsletter/roles', 'jira-help-tips/feature/help-tip', 'atlassian/libs/underscore-1.8.3', 'jquery'], function (analytics, formatter, Meta, roles, HelpTip, _, $) {

    // taken from setup-mac-util.js
    // http://www.w3.org/TR/html5/forms.html#valid-e-mail-address without &
    var emailRegex = /^[a-zA-Z0-9.!#$%'*+/=?^_`{|}~-]+@[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*$/;

    function validateEmail(email) {
        if (email.length > 255) {
            return false;
        }
        return emailRegex.test(email);
    }

    function checksum(value) {
        return _.reduce(value, function (r, ch) {
            r = (r << 5) - r + ch.charCodeAt(0);
            return r & r;
        }, 0);
    }

    function omitUndefinedValues(obj) {
        return _.omit(obj, _.isUndefined);
    }

    return {
        render: function render(newsletterFormDetails) {
            var signupTipDelay = _.random(10000);
            setTimeout(function () {
                this.doRender(_.extend(newsletterFormDetails, { signupTipDelay: signupTipDelay }));
            }.bind(this), signupTipDelay);
        },

        doRender: function doRender(newsletterFormDetails) {
            var defaults = {
                anchor: "#user-options",
                isSequence: false,
                showCloseButton: false
            };

            defaults.id = newsletterFormDetails.signupId;
            defaults.title = newsletterFormDetails.signupTitle;
            defaults.bodyHtml = JIRA.Templates.newsletterSignupTip({
                userEmail: newsletterFormDetails.userEmail,
                description: newsletterFormDetails.signupDescription,
                roles: roles.getRoles()
            });

            var tip = new HelpTip(defaults);
            if (tip.isDismissed()) {
                return;
            }

            var getFlagsVisible = function getFlagsVisible() {
                return $("#aui-flag-container").find(".aui-flag").filter(":not([aria-hidden=true]):visible").length;
            };
            if (getFlagsVisible()) {
                analytics.send({
                    name: "jira.newsletter.signuptip.notshown",
                    data: {
                        flagsVisible: getFlagsVisible(),
                        signupTipDelay: newsletterFormDetails.signupTipDelay
                    }
                });
                return;
            }

            tip.show();

            analytics.send({ name: "jira.newsletter.signuptip.shown" });

            var $body = $("body");
            $body.on("submit", "form.aui.insiders-signup-form", function (e) {
                e.preventDefault();

                var $form = $(this);
                $form.find(".error").remove();

                var $emailInput = $form.find("#jira-newsletter-user-email");
                var $roleSelect = $form.find("#jira-newsletter-role");
                var $optInCheckbox = $form.find("#jira-newsletter-opt-in");
                var email = $emailInput.val();
                var role = $roleSelect.val();
                var optIn = $optInCheckbox.prop('checked');

                //Since otherRoleName is whitelisted is good to not allowed sending too much data here
                var otherRoleName = role === 'other' ? $form.find("#jira-newsletter-other-role-name").val().substring(0, 100) : undefined;
                var emailIsValid = validateEmail(email);
                var optInSelected = optIn === true;

                var formIsValid = emailIsValid && optInSelected;

                if (formIsValid) {
                    $.ajax({
                        type: 'POST',
                        url: formatter.format(newsletterFormDetails.formUrl, encodeURI(email)),
                        dataType: 'json',
                        contentType: 'application/json',
                        data: JSON.stringify({
                            role: role,
                            otherRoleName: otherRoleName,
                            lang: Meta.getLanguage()
                        })
                    }).success(function () {
                        analytics.send({
                            name: "jira.newsletter.signuptip.submitted",
                            data: omitUndefinedValues({
                                role: role,
                                otherRoleName: otherRoleName,
                                lang: Meta.getLanguage(),
                                // https://extranet.atlassian.com/x/aDhGvw
                                checksum: checksum(email.split('@')[0]),
                                flagsVisible: getFlagsVisible(),
                                signupTipDelay: newsletterFormDetails.signupTipDelay
                            })
                        });
                    }).error(function (xhr) {
                        analytics.send({
                            name: "jira.newsletter.signuptip.error",
                            data: {
                                statusCode: xhr.status
                            }
                        });
                    });

                    if (newsletterFormDetails.userEmail !== email) {
                        analytics.send({ name: "jira.newsletter.signuptip.email.changed" });
                    }

                    tip.dismiss("newslettersubscribed");
                } else {
                    if (!emailIsValid) {
                        analytics.send({
                            name: "jira.newsletter.signuptip.email.validationerror",
                            data: {
                                flagsVisible: getFlagsVisible()
                            }
                        });
                        $emailInput.after(aui.form.fieldError({ message: formatter.I18n.getText('newsletter.signup.tip.error.email') })); // eslint-disable-line no-undef
                    }
                    if (!optInSelected) {
                        analytics.send({
                            name: "jira.newsletter.signuptip.opt-in.validationerror",
                            data: {
                                flagsVisible: getFlagsVisible()
                            }
                        });
                        $optInCheckbox.parent().after(aui.form.fieldError({ message: formatter.I18n.getText('newsletter.signup.tip.error.opt-in') })); // eslint-disable-line no-undef
                    }
                }
            });

            $body.on("click", "form.aui.insiders-signup-form a.cancel", function (e) {
                e.preventDefault();
                analytics.send({
                    name: "jira.newsletter.signuptip.dismissed",
                    data: {
                        flagsVisible: getFlagsVisible(),
                        signupTipDelay: newsletterFormDetails.signupTipDelay
                    }
                });
                tip.dismiss("newslettercancelled");
            });

            $body.on("change", "form.aui.insiders-signup-form", function (e) {
                var $form = $(this);
                var $roleSelect = $form.find("#jira-newsletter-role");
                var $otherRoleName = $form.find("#jira-newsletter-other-role-name");

                $otherRoleName.parent().closest('div').toggleClass('hidden', $roleSelect.val() !== 'other');
            });
        }
    };
});