define("jira/projects/test-utils/mockutils", ['require'], function (require) {
    "use strict";

    var _ = require('jira/projectissuenavigator/libs/underscore');

    return {
        /**
         * Requires a AMD module using a list of custom modules for dependencies.
         *
         * This method can be used to inject mocked dependencies into an AMD module. Example:
         *
         * ```
         * var mod = requireWithMocks('mod', {
         *    'dep1': function() {},
         *    'dep2': function() {}
         * }
         * ```
         *
         * will require the module 'mod' using two empty functions as the dependencies 'dep1' and 'dep2'
         *
         * @param {string} moduleName Module to load
         * @param {Object} mocks List of dependenies to inject
         * @returns {*} The module
         */
        requireWithMocks: function(moduleName, mocks) {
            var context = AJS.test.mockableModuleContext();

            _.each(mocks, function(mock, name){
                context.mock(name, mock);
            });

            return context.require(moduleName);
        },

        /**
         * Returns a constructor that will generate an instance with all methods spied.
         *
         * This can be used when a class is constructing an object privately, but we want to be able
         * to spy on the constructor *and* on the generated instance. Example:
         *
         * var MyView = spyAll(sinon, View);
         *
         * function run() {
         *    var view = new MyView();
         *    view.render();
         * }
         * run()
         *
         * MyView.called === true //true
         * MyView.getInstance().render.called === true //true
         *
         * The returned constructor will have the method 'getInstance()', that will return the latest
         * instance generated by such constructor. Example:
         *
         * ```
         * var A = spyAll(sinon, require('A'));
         * var a = new A();
         * A.getInstance() === a //true
         * ```
         *
         * @param sinon Instance of sinon (or sinon's sandbox) to use for the stubs/spies
         * @param ctor Class to spy
         * @returns {function} Function that will generate an spied instance of 'ctor' when invoked with new'
         */
        spyAll: function(sinon, ctor) {
            var instance;
            function F() {
                ctor.apply(this, arguments);

                // Store the instance
                instance = this;
            }
            F.prototype = ctor.prototype;
            F.getInstance = function() {
                return instance;
            };

            // Spy all the functions in the original prototype
            for (var prop in ctor.prototype) {
                if (typeof ctor.prototype[prop] === "function") {
                    sinon.spy(ctor.prototype, prop);
                }
            }

            return sinon.spy(F);
        }
    };
});
