/**
 * Base configuration for building parameterised tests for js/services/API.js.
 *
 * This API behaves slightly differently depending on where it's used (which issue viewing component):
 * - The fullscreen View Issue page at /browse/${ISSUE_KEY};
 * - The fullscreen in PIN or GIN; and,
 * - The split-screen details layout in PIN or GIN.
 *
 * Behaviour changes again based on whether or not Inline Editing is enabled or disabled by the administrator. We
 * generate tests for all combinations of component and inline editing state (including inline editing setting =
 * undefined for completeness).
 *
 * Originally, these were all generated in js/services/API-test.js, but they were taking so long to run that the web
 * driver that run it was timing out and failing. Instead of just increasing the timeout, I've moved some combinations
 * into their own files (all components get their own files, but generate all their inline editing combinations).
 *
 * The components listed above are tested in:
 * - js/services/BrowseIssueViewAPI-test.js;
 * - js/services/IssueViewAPI-test.js; and,
 * - js/services/IssueSearchAPI-test.js.
 *
 * The old js/services/API-test.js module contains simple, more general tests (unparameterised), but uses much of the
 * same setup.
 */
define("jira/projects/test-utils/issue-api-test", [
    "jira/projects/test-utils/projectissuenavigatorworldmocker",
    "jira/projectissuenavigator/services/api",
    "jira/flag",
    "jira/projectissuenavigator/libs/underscore",
    "wrm/data",
    "jquery"
], function(World, API, JIRAFlag, _, WrmData, jQuery) {
    "use strict";

    var INLINE_EDIT_WRM_DATA_KEY = 'com.atlassian.jira.jira-issue-nav-components:inline-edit-enabled';

    var ISSUES = [
        {
            id: 1,
            key: 'DEMO-1',
            summary: 'The first issue in the project',
            status: "Open",
            type: {
                name: 'bug',
                description: 'A bug in your application',
                icon: 'iconUrl.png'
            }
        },
        {
            id: 2,
            key: 'DEMO-2',
            summary: 'The second issue in the project',
            status: "Open",
            type: {
                name: 'bug',
                description: 'A bug in your application',
                icon: 'iconUrl.png'
            }
        },
        {
            id: 3,
            key: 'DEMO-3',
            summary: 'The third issue in the project',
            status: "Open",
            type: {
                name: 'bug',
                description: 'A bug in your application',
                icon: 'iconUrl.png'
            }
        },
        {
            id: 4,
            key: 'DEMO-4',
            summary: 'The fourth issue in the project',
            status: "Open",
            type: {
                name: 'bug',
                description: 'A bug in your application',
                icon: 'iconUrl.png'
            }
        }
    ];

    var BASE_MODULE_CONFIGURATION = {
        setup: function() {
            var sandbox = sinon.sandbox.create();
            this.world = new World(sandbox);

            this.oldAPISidebar = JIRA.API.Sidebar;
            JIRA.API.Sidebar = {
                getAUISidebar: function() {
                    return {};
                }
            };

            jQuery("body").css("overflow-y", "auto");
            API.init();
        },
        teardown: function() {
            this.world.restore();

            // The IssueEditor hooks on onbeforeunload to show warnings, make sure
            // we clean it up when the test is done
            window.onbeforeunload = null;

            // SidebarIssueSearch hacks the <body> to prevent scrolling, clean it up
            jQuery("body").removeClass("page-type-split").removeClass("page-issue-navigator");

            JIRA.API.Sidebar = this.oldAPISidebar;
        },
        setInlineEditParam: function(enabled) {
            WrmData.claim
                .withArgs(INLINE_EDIT_WRM_DATA_KEY)
                .returns(enabled);
        },
        worldWithDefaultData: function(data) {
            return this.world
                .withData(data || {
                        issue: 'DEMO-2',
                        filter: 'allissues',
                        issues: ISSUES,
                        editableFields: [{
                            id: "summary",
                            editHtml: '<div><input data-id="2" type="text" name="summary"></div>'
                        }]
                    });
        },
        loadIssueSearchWithDefaultData: function(data) {
            this.worldWithDefaultData(data)
                .build()
                .waitForIssueSearch();
        },
        loadIssueViewWithDefaultData: function(data) {
            this.loadIssueSearchWithDefaultData(data);
            this.world
                .goToIssueView()
                .waitForEditor();
        },
        assertIsInIssueView: function() {
            ok(this.world.projectIssueNavigator.issueView);
            ok(!this.world.projectIssueNavigator.issueSearch);
        },
        assertIsInIssueSearch: function() {
            ok(!this.world.projectIssueNavigator.issueView);
            ok(this.world.projectIssueNavigator.issueSearch);
        }
    };

    var INLINE_EDIT_CASES = [
        {enabled: true},
        {enabled: false},
        {enabled: undefined}
    ];

    function makeTestsForComponent(component) {
        INLINE_EDIT_CASES.forEach(function(inlineEditCase) {
            var moduleName = 'jira/projectissuenavigator/services/api [' + component.name + ', inline edit ' + String(inlineEditCase.enabled) + ']';
            module(moduleName, _.extend({}, BASE_MODULE_CONFIGURATION, component, {
                setup: function() {
                    BASE_MODULE_CONFIGURATION.setup.call(this);
                    BASE_MODULE_CONFIGURATION.setInlineEditParam.call(this, inlineEditCase.enabled);
                }
            }));

            test("JIRA.API.IssueSearch.nextIssue() - It selects the next issue", function() {
                this.load();

                JIRA.API.IssueSearch.nextIssue();
                this.world.waitForPageLoad();
                this.world.waitForEditor();

                this.assertComponentIsVisible();
                this.assertIssueAfterNavigation("DEMO-3");
            });

            test("JIRA.API.IssueSearch.nextIssue() - It start recording browser metrics", function() {
                this.load();

                JIRA.API.IssueSearch.nextIssue();
                this.world.waitForPageLoad();
                this.world.waitForEditor();

                this.assertComponentIsVisible();
                this.assertMetricsCalledAfterNavigation(this.world.Metrics.startIssueSearch);
            });

            test("JIRA.API.IssueSearch.previousIssue() - It selects the previous issue", function() {
                this.load();

                JIRA.API.IssueSearch.previousIssue();
                this.world.waitForPageLoad();
                this.world.waitForEditor();

                this.assertComponentIsVisible();
                this.assertIssueAfterNavigation("DEMO-1");
            });

            test("JIRA.API.IssueSearch.previousIssue() - It start recording browser metrics", function() {
                this.load();

                JIRA.API.IssueSearch.previousIssue();
                this.world.waitForPageLoad();
                this.world.waitForEditor();

                this.assertComponentIsVisible();
                this.assertMetricsCalledAfterNavigation(this.world.Metrics.startIssueSearch);
            });

            test("JIRA.API.IssueSearch.getActiveIssueId() - It retrieves the active issue id", function() {
                this.load();

                var issueId = JIRA.API.IssueSearch.getActiveIssueId();

                this.assertComponentIsVisible();
                equal(issueId, 2);
            });

            test("JIRA.API.IssueSearch.getActiveIssueKey() - It retrieves the active issue key", function() {
                this.load();

                var issueId = JIRA.API.IssueSearch.getActiveIssueKey();

                this.assertComponentIsVisible();
                equal(issueId, "DEMO-2");
            });

            test("JIRA.API.IssueSearch.refreshActiveIssue() - It refreshes the active issue", function() {
                this.load();

                this.world.withUpdatedIssue("DEMO-2", {
                    summary: "This is the new summary"
                });

                JIRA.API.IssueSearch.refreshActiveIssue();
                this.world.waitForEditor();

                this.assertComponentIsVisible();
                equal(this.getSummary(), "This is the new summary");
            });

            test("JIRA.API.IssueSearch.refreshActiveIssue() - It returns a promise", function() {
                this.load();

                this.world.withUpdatedIssue("DEMO-2", {
                    summary: "This is the new summary"
                });

                var promise = JIRA.API.IssueSearch.refreshActiveIssue();
                this.world.waitForEditor();

                this.assertComponentIsVisible();
                equal(typeof promise.done, "function", "It contains the done method");
                equal(typeof promise.fail, "function", "It contains the fail method");
                equal(typeof promise.then, "function", "It contains the then method");
                equal(typeof promise.always, "function", "It contains the always method");
                equal(typeof promise.pipe, "function", "It contains the pipe method");
            });

            test("JIRA.API.IssueSearch.refreshActiveIssue() - It start recording browser metrics", function() {
                this.load();

                this.world.withUpdatedIssue("DEMO-2", {
                    summary: "This is the new summary"
                });

                JIRA.API.IssueSearch.refreshActiveIssue();
                this.world.waitForEditor();

                this.assertComponentIsVisible();
                this.assertMetricsCalledAfterNavigation(this.getBrowserMetricsFunction());
            });

            test("JIRA.API.IssueSearch.refreshIssue() - It refreshes any issue based on the issueId", function() {
                this.load();

                this.world.withUpdatedIssue("DEMO-2", {
                    summary: "This is the new summary"
                });

                JIRA.API.IssueSearch.refreshIssue(2);
                this.world.waitForEditor();

                this.assertComponentIsVisible();
                equal(this.getSummary(), "This is the new summary");
            });

            test("JIRA.API.IssueSearch.refreshIssue() - It returns a promise", function() {
                this.load();

                this.world.withUpdatedIssue("DEMO-2", {
                    summary: "This is the new summary"
                });

                var promise = JIRA.API.IssueSearch.refreshIssue(2);
                this.world.waitForEditor();

                this.assertComponentIsVisible();
                equal(typeof promise.done, "function", "It contains the done method");
                equal(typeof promise.fail, "function", "It contains the fail method");
                equal(typeof promise.then, "function", "It contains the then method");
                equal(typeof promise.always, "function", "It contains the always method");
                equal(typeof promise.pipe, "function", "It contains the pipe method");
            });

            test("JIRA.API.IssueSearch.refreshIssue() - It start recording browser metrics", function() {
                this.load();

                this.world.withUpdatedIssue("DEMO-2", {
                    summary: "This is the new summary"
                });
                this.world.Metrics.startIssueSearch.reset();

                JIRA.API.IssueSearch.refreshIssue(2);
                this.world.waitForEditor();

                this.assertComponentIsVisible();
                this.assertMetricsCalledAfterNavigation(this.getBrowserMetricsFunction());
            });

            test("JIRA.API.IssueSearch.editField() - It edits a field in the active issue", function() {
                this.load();

                this.world.waitForEditorFields();

                JIRA.API.IssueSearch.editField('summary');

                this.assertComponentIsVisible();
                ok(this.world.dialogEl.indexOf("<div><input data-id='2' type='text' name='summary'/></div>" > -1));
            });

            test("JIRA.API.IssueSearch.getFields() - It get the fields from the active issue", function() {
                this.load();

                this.world.waitForEditorFields();

                var fields = JIRA.API.IssueSearch.getFields();

                this.assertComponentIsVisible();
                equal(fields.length, 1);
                equal(jQuery(fields.get('summary').get("editHtml")).find("input[data-id=2][type=text][name=summary]").length, 1);
            });

            test("JIRA.API.IssueSearch.isSaving() - It detects when there are saves in progress", function() {
                this.load();

                this.world.waitForEditorFields();

                var fields = JIRA.API.IssueSearch.getFields();
                fields.get('summary').edit();
                fields.get('summary').save();

                this.assertComponentIsVisible();
                ok(JIRA.API.IssueSearch.isSaving());
            });

            test("JIRA.API.IssueSearch.isLoading() - It detects when the IssueEditor is loading", function() {
                this.startLoading();

                ok(JIRA.API.IssueSearch.isLoading());

                this.world.waitForEditor();

                this.assertComponentIsVisible();
                ok(!JIRA.API.IssueSearch.isLoading());
            });

            test("JIRA.API.IssueSearch.hasAccessibleIssue() - It detects if there is an issue loaded when there are results", function() {
                this.load();

                this.world.waitForEditor();

                this.assertComponentIsVisible();
                ok(JIRA.API.IssueSearch.hasAccessibleIssue());
            });

            test("JIRA.API.IssueSearch.hasAccessibleIssue() - It detects that there is not an issue loaded when there are no results", function() {
                this.load({
                    issue: 'DEMO-2',
                    filter: 'allissues',
                    issues: []
                });

                ok(!JIRA.API.IssueSearch.hasAccessibleIssue());
            });

            test("JIRA.API.IssueSearch.hasAccessibleIssue() - It detects that there is not an issue loaded when the issue has been deleted", function() {
                this.load();
                this.world.withUpdatedIssue("DEMO-2", {
                    inaccessible: true
                });

                this.assertComponentIsVisible();
                JIRA.API.IssueSearch.refreshActiveIssue();
                this.world.waitForEditor();

                this.assertComponentIsVisible();
                ok(!JIRA.API.IssueSearch.hasAccessibleIssue());
            });

            test("JIRA.API.IssueSearch.showLoadError() - It shows the 'loading error' flag", function() {
                this.load();

                JIRA.API.IssueSearch.showLoadError();

                this.assertComponentIsVisible();
                sinon.assert.calledOnce(this.world.JIRAFlag.showErrorMsg);
                equal(this.world.JIRAFlag.showErrorMsg.lastCall.args[0], '', "Title");
                equal(this.world.JIRAFlag.showErrorMsg.lastCall.args[1], JIRA.Components.IssueViewer.Templates.Body.errorsLoading(), "Body");
            });

            test("JIRA.API.IssueSearch.waitForSavesToComplete() - It waits for saves to complete using a promise, resolved if there are no pending saves", function() {
                this.load();
                var onSave = this.spy();

                JIRA.API.IssueSearch.waitForSavesToComplete().done(onSave);

                this.assertComponentIsVisible();
                sinon.assert.calledOnce(onSave);
            });

            test("JIRA.API.IssueSearch.waitForSavesToComplete() -  It waits for saves to complete using a promise, resolved when all the saves are completed", function() {
                this.load();
                this.world.waitForEditorFields();
                var onSave = this.spy();

                var fields = JIRA.API.IssueSearch.getFields();
                fields.get('summary').edit();
                fields.get('summary').save();
                JIRA.API.IssueSearch.waitForSavesToComplete().done(onSave);

                this.assertComponentIsVisible();
                sinon.assert.notCalled(onSave);

                this.world.waitForEditorSave();

                this.assertComponentIsVisible();
                sinon.assert.calledOnce(onSave);
            });

            test("JIRA.API.IssueSearch.focusSearch() - Does nothing", function() {
                this.load();

                JIRA.API.IssueSearch.focusSearch();

                this.assertComponentIsVisible();
            });

            test("JIRA.API.IssueSearch.toggleFilterPanel() - Does nothing", function() {
                this.load();

                JIRA.API.IssueSearch.toggleFilterPanel();

                this.assertComponentIsVisible();
            });

            test("JIRA.API.IssueSearch.isQueryValid() - All queries are valid", function() {
                this.load();

                this.assertComponentIsVisible();
                ok(JIRA.API.IssueSearch.isQueryValid());
            });

            test("JIRA.API.IssueSearch.isFullScreenIssueVisible() - It detects if it is in IssueView mode", function() {
                this.load();

                this.assertComponentIsVisible();
                this.assertFullScreenIssueVisible();
            });

            test("JIRA.API.IssueSearch.switchLayouts() in IssueSearch - It does nothing", function() {
                this.load();

                JIRA.API.IssueSearch.switchLayouts();

                this.assertComponentIsVisible();
            });

            test("JIRA.API.IssueSearch.openFocusShifter() in IssueSearch - It opens the focus shifter", function() {
                this.load();

                JIRA.API.IssueSearch.openFocusShifter();

                this.assertComponentIsVisible();
                sinon.assert.calledOnce(this.world.FocusShifter.prototype.show);
            });
        });
    }

    return {
        issuesFixture: ISSUES,
        baseModuleConfiguration: BASE_MODULE_CONFIGURATION,
        makeTestsForComponent: makeTestsForComponent
    };
});
