define("jira/projects/test-utils/fakeserver", ['require'], function (require) {
    "use strict";

    var wrmContextPath = require('wrm/context-path')();
    var _ = require('jira/projectissuenavigator/libs/underscore');

    /**
     * This method transforms a request body like 'a=1&b=2&b=3' into an object like
     * {
     *    a: 1
     *    b: [2, 3]
     * }
     */
    function parseRequestBody(request) {

        var variables = request.requestBody.split("&");

        var result = {};
        _.each(variables, function(chunk) {
            var chunks = chunk.split("=");
            var key = chunks[0];
            var value = chunks[1] || "";

            if (key in result) {
                result[key] = [].concat(result[key], value);
            }else{
                result[key] = value;
            }
        });
        return result;
    }

    /**
     * Creates a fakeServer using either a sinon "native" object or a sinon sandbox.
     * We need this methods because the API to create a server is different in each case.
     *
     * @param {Object} sinon Sinon main object or a sinon Sandbox/
     * @returns {Object} A sinon fake server.
     */
    function createServer(sinon) {
        var server;
        if (sinon.useFakeServer) {
            sinon.useFakeServer();
            server = sinon.server;
        } else {
            server = sinon.fakeServer.create();
        }
        return server;
    }

    function createIssueSearchInterceptor(options) {
        var issues = options.issues;
        var pageSize = options.pageSize;
        var totalIssues = options.totalIssues;

        return function(request) {
            var response = {
                "issueTable": {
                    "displayed": pageSize,
                    "end": pageSize,
                    "issueIds": _.pluck(issues, "id"),
                    "issueKeys": _.pluck(issues, "key"),
                    "page": 0,
                    "pageSize": pageSize,
                    "startIndex": 0,
                    "table": issues.slice(0, pageSize),
                    "total": totalIssues
                }
            };

            request.respond(200, { "Content-Type": "application/json" }, JSON.stringify(response));
        };
    }

    function createIssueStableSearchInterceptor(options) {
        var issues = options.issues;

        return function(request) {
            var requestParameteres = parseRequestBody(request);

            // Note: the pagination state for a stable response is broken in the actual server implementation. This
            // mocked has the same problems (e.g. total == pageSize, page always 0...) for the sake of testing.
            requestParameteres.id = [].concat(requestParameteres.id);
            var response = {
                "issueTable": {
                    "displayed": requestParameteres.id.length,
                    "end": requestParameteres.id.length,
                    "page": 0,
                    "pageSize": requestParameteres.id.length,
                    "startIndex": 0,
                    "table": _.map(requestParameteres.id, function(id) {return _.findWhere(issues, {id: Number(id)}); } ),
                    "total": requestParameteres.id.length
                }
            };

            request.respond(200, { "Content-Type": "application/json" }, JSON.stringify(response));
        };
    }

    function createIssueViewLoadInterceptor(options) {
        var issues = options.issues;
        return function (request, issueKey) {
            if (request.method === 'POST') {
                issueKey = parseRequestBody(request).issueKey;
            }
            var issue = _.findWhere(issues, {key: issueKey});
            if (!issue || issue.inaccessible) {
                request.respond(404, {"Content-Type": "application/json"}, JSON.stringify({
                    "fields": [],
                    "errorCollection": {"errorMessages": ["Issue Does Not Exist"], "errors": {}}
                }));
            } else {
                var response = {
                    issue: _.extend({}, issue, {
                        operations: {
                            linkGroups: []
                        }
                    }),
                    panels: {
                        leftPanels: [],
                        rightPanels: [],
                        infoPanels: []
                    },
                    fields: {}
                };
                request.respond(200, {"Content-Type": "application/json"}, JSON.stringify(response));
            }
        };
    }

    function createIssueEditLoadInterceptor(options) {
        var issues = options.issues;
        return function(request, issueId) {
            var issue = _.findWhere(issues, {id: Number(issueId)});

            var response = { fields: issue.editableFields };

            request.respond(200, { "Content-Type": "application/json" }, JSON.stringify(response));
        };
    }

    function createIssueLoadInterceptor(options) {
        var issues = options.issues;
        return function(request, issueId) {
            var issue = _.findWhere(issues, {id: Number(issueId)});
            var response = {
                id: issue.id,
                key: issue.key,
                fields: {
                    summary: issue.summary,
                    status: {
                        name: issue.status
                    },
                    issuetype: {
                        description: issue.type.description,
                        iconUrl: issue.type.icon,
                        name: issue.type.name
                    }
                }
            };

            request.respond(200, { "Content-Type": "application/json" }, JSON.stringify(response));
        };
    }

    function createIssueSaveInterceptor(options) {
        var issues = options.issues;
        return function(request) {
            var requestParameteres = parseRequestBody(request);

            var issue = _.findWhere(issues, {id: Number(requestParameteres.issueId)});
            var response = {
                id: issue.id,
                key: issue.key,
                fields: {
                    summary: requestParameteres.summary,
                    status: {
                        name: issue.status
                    },
                    issuetype: {
                        description: issue.type.description,
                        iconUrl: issue.type.icon,
                        name: issue.type.name
                    }
                }
            };

            request.respond(200, { "Content-Type": "application/json" }, JSON.stringify(response));
        };
    }

    return {
        /**
         * Creates a sinon FakeServer that mimics the response of all the calls perform by the DetailsLayout component:
         *   - Initial search
         *   - Load a page
         *   - Load the issue editor in read-only mode)
         *   - Load the inline-edit fields for the issue editor
         *   - Save a field modification
         *   - Refresh a single issues
         *
         * It receives a list of issues to work with. Each issue have the format:
         *   {
         *       id: 3,
         *       key: 'DEMO-3',
         *       summary: 'The third issue in the project',
         *       status: "Open",
         *       type: {
         *           name: 'bug',
         *           description: 'A bug in your application',
         *           icon: 'iconUrl.png'
         *       }
         *   }
         *
         * Please, note that this is the bare minimum implementation for DetailsLayout to not break, this is not the
         * full implementation of the server responses.
         *
         * @param {Object} sinon Sinon implementation used to create the fake server
         * @param {Array} issues List of issues.
         * @return {Object} Sinon's Fake Server
         */
        create: function(sinon, issues) {
            var server = createServer(sinon);

            server.respondWith("POST", new RegExp(wrmContextPath + "/rest/issueNav/1/issueTable$"), createIssueSearchInterceptor({
                issues: issues,
                pageSize: 2,
                totalIssues: issues.length
            }));

            server.respondWith("POST", new RegExp(wrmContextPath + "/rest/issueNav/1/issueTable/stable$"), createIssueStableSearchInterceptor({
                issues: issues
            }));

            server.respondWith("GET", new RegExp(wrmContextPath + "/secure/AjaxIssueAction!default\\.jspa.*issueKey=([A-Za-z0-9-]*)"), createIssueViewLoadInterceptor({
                issues: issues
            }));

            server.respondWith("POST", new RegExp(wrmContextPath + "/secure/AjaxIssueAction!default\\.jspa"), createIssueViewLoadInterceptor({
                issues: issues
            }));

            server.respondWith("GET", new RegExp(wrmContextPath + "/secure/AjaxIssueEditAction!default\\.jspa.*issueId=([0-9]*)"), createIssueEditLoadInterceptor({
                issues: issues
            }));

            server.respondWith("GET", new RegExp(wrmContextPath + "/rest/api/2/issue/([0-9]*)"), createIssueLoadInterceptor({
                issues: issues
            }));

            server.respondWith("POST", new RegExp(wrmContextPath + "/secure/AjaxIssueAction.jspa"), createIssueSaveInterceptor({
                issues: issues
            }));

            return server;
        }
    };
});
