define("jira/projectissuenavigator/services/router", ['require'], function (require) {
    "use strict";

    var Backbone = require('jira/projectissuenavigator/libs/backbone');
    var URLHelper = require('jira/projectissuenavigator/services/urlhelper');
    var Browser = require('jira/projectissuenavigator/services/browser');
    var Meta = require('jira/util/data/meta');
    var contextPath = require('wrm/context-path')();
    var URI = require('jira/projectissuenavigator/libs/uri');
    var _ = require('jira/projectissuenavigator/libs/underscore');

    return Backbone.Router.extend({
        initialize: function() {
            this._isFirstLoad = true;

            // Match any route. We will extract the parts (issue, filter and orderby) from the URL manually.
            this.route(/.*/, function() {
                var state = URLHelper.extractStateFromCurrentUrl();
                var payload = {
                    issue: state.issue,
                    filter: state.filter,
                    orderby: state.orderby,
                    startIndex: state.startIndex,
                    project: state.project
                };

                var trigger = (function (event, initialEvent) {
                    if (this._isFirstLoad) {
                        this._isFirstLoad = false;
                        this.trigger(initialEvent, payload);
                    } else {
                        this.trigger(event, payload);
                    }
                }.bind(this));
                if (URLHelper.isBrowseUrl()) {
                    trigger("route:browse", "route:browse:initial");
                } else {
                    trigger("route:issues", "route:issues:initial");
                }
            });
        },

        /**
         * Generates a URL from the passed state and performs PushState.
         *
         * @param {Object} options
         * @param {string} options.issue Issue key, e.g 'KEY-123'
         * @param {string} [options.project] Project key, e.g. 'KEY'
         * @param {string} [options.filter] Filter ID, can be a string like 'allissues' or a number like 10000
         * @param {string} [options.orderby] Order definition, can have multiple columns, separated by a comma, e.g. 'created DESC'
         * @param {boolean} [options.isIssueView] True if this should be a '/browse/KEY-123' URL
         * @param {boolean} [options.replaceHistory] True if PushState should not add a history entry but instead replace the current one
         * @param {boolean} [options.keepParameters] True if query parameters and the hash of the current URL should be preserved
         */
        updateURL: function(options) {
            options = options || {};
            // Always update the Permalink, even if the new URL is the same as the previous URL. This way we cover
            // the case where the Permalink is not initially set to the URL of the page.
            this._updatePermalinkState(options);

            var urlOptions = {
                issue: options.issue,
                project: options.project,
                filter: options.filter,
                orderby: options.orderby
            };

            var url = options.isIssueView ? URLHelper.generateBrowseUrl(urlOptions) : URLHelper.generateUrl(urlOptions);
            var newUri = new URI(url);
            // Generate the current URL with just the path, query and fragment.
            var existingFullUri = new URI(Browser.getCurrentUrl());
            var existingUri = new URI({
                path: existingFullUri.path(),
                query: existingFullUri.query(),
                fragment: existingFullUri.fragment()
            });

            // Do nothing if the URL is the same. This simulates the behaviour of backbone.queryparams.js
            if (newUri.equals(existingUri)) return;

            if (options.keepParameters) {
                newUri.search(_.extend(existingUri.search(true), newUri.search(true)));
                newUri.hash(existingUri.hash());
                url = newUri.toString();
            }

            this.navigate(url, {trigger: false, replace: !!options.replaceHistory });
        },

        goTo: function(options) {
            options = options || {};
            Browser.locationAssign(URLHelper.generateUrl({
                project: options.project,
                issue: options.issue,
                filter: options.filter,
                orderby: options.orderby
            }));
        },

        _getBaseurl: function () {
            return URI.build(
                _.extend(
                    _.pick(URI.parse(Browser.getCurrentUrl()), 'protocol', 'hostname', 'port', 'username', 'password'),
                    { path: contextPath }
                )
            );
        },

        _getServerBaseurl: function () {
            var jiraBaseUrl = Meta.get("jira-base-url");

            if (jiraBaseUrl) {
                return jiraBaseUrl;
            }

            return this._getBaseurl();
        },


        _updatePermalinkState: function(options) {
            if (options.issue) {
                Meta.set("viewissue-permlink",
                    //When the contextPath is empty, a double bar appears. URI.normalize fixes the problem.
                    new URI(this._getServerBaseurl() + "/browse/" + options.issue).normalize().toString()
                );
            }
        },

        start: function() {
            var root = URLHelper.generateRootUrl();

            // If we are using a pushState url (/KEY-123) in a non-pushState browser,
            // transform it to be hashed (#KEY-123). In other words, this is for convert
            // Chrome URLs into IE9 URLs.
            // The opposite case (IE9 URL to Chrome URL) is already handled nicely by Backbone.
            var browserSupportsPushState = Browser.hasPushState();
            var appUrl = URLHelper.getApplicationUrl();
            var urlIsForPushStateBrowsers = appUrl && !URLHelper.isHashUrl();
            if (!browserSupportsPushState && urlIsForPushStateBrowsers) {
                Browser.locationReplace(root + appUrl.replace(/^\//, "#"));
                return;
            }

            Backbone.history.start({
                root: "",
                pushState: true
            });
        }
    });
});
