define("jira/projectissuenavigator/services/filters", ['require'], function (require) {
    "use strict";

    var wrmContextPath = require('wrm/context-path')();
    var _ = require('jira/projectissuenavigator/libs/underscore');
    var WrmData = require('wrm/data');
    var AJSHelper = require('jira/projectissuenavigator/libs/ajshelper');
    var ProjectsAPI = require('jira/projectissuenavigator/apis/projects');

    var genericFilters;
    var projectFilters;
    var canManageFilters;
    var pid;
    var defaultFilterPriority;

    function getFiltersAsMap(filters) {
        return _.indexBy(_.flatten(filters), 'id');
    }

    return {
        initialize: function() {
            genericFilters = WrmData.claim("com.atlassian.jira.jira-projects-issue-navigator:generic-filters") || [];
            projectFilters = WrmData.claim("com.atlassian.jira.jira-projects-issue-navigator:project-filters") || [];
            canManageFilters = WrmData.claim("com.atlassian.jira.jira-projects-issue-navigator:can-manage-filters");
            defaultFilterPriority = WrmData.claim("com.atlassian.jira.jira-projects-issue-navigator:default-filter-priority") || [ this.getAllIssuesFilterId() ];
            pid = ProjectsAPI.getCurrentProjectId();
        },

        getExtraLinks: function() {
            return [{
                id: 'manage',
                label: AJS.I18n.getText("common.concepts.managefilters"),
                link: wrmContextPath + "/secure/ManageFilters.jspa?searchShareType=project&projectShare=" + pid + "&Search=Search&filterView=search"
            }];
        },

        getAvailableFilters: function(includeLinkToManageFilters) {
            var filters = [];

            if (genericFilters && genericFilters.length) {
                filters.push(genericFilters);
            }

            if (projectFilters && projectFilters.length) {
                filters.push(projectFilters);

                if (includeLinkToManageFilters && canManageFilters) {
                    filters.push(this.getExtraLinks());
                }
            }

            return filters;
        },

        getDefaultFilterId: function() {
            // find first filter that's available
            return _.find(defaultFilterPriority, _.bind(function (id) {
                return !!_.findWhere(_.flatten(this.getAvailableFilters()), { id: id });
            }, this));
        },

        getJQL: function(filterId, projectKey, orderby) {
            var filter = _.findWhere(_.flatten(this.getAvailableFilters()), { id: filterId });
            if (filter) {
                return AJSHelper.format(filter.jql, projectKey, orderby || filter.defaultOrderby);
            }
        },

        getAllIssuesFilterId: function() {
            return 'allissues';
        },

        sanitiseFilter: function(filterId) {
            if (_.some(_.flatten(this.getAvailableFilters()), function (availableFilter) {return availableFilter.id === filterId;})) {
                return filterId;
            }
            return this.getDefaultFilterId();
        },

        filterSupportsInlineIssueCreate: function(filterId) {
            var filters = getFiltersAsMap(this.getAvailableFilters(false));
            var filter = filters[filterId];

            return Boolean(filter && filter.supportsInlineIssueCreate);
        }
    };
});
